import type { PaginationProps } from '../types/pagination';
import type { BasicTableProps } from '../types/table';
import { computed, unref, ref, ComputedRef, watch } from 'vue';
import { LeftOutlined, RightOutlined } from '@ant-design/icons-vue';
import { isBoolean } from '/@/utils/is';
import { PAGE_SIZE, PAGE_SIZE_OPTIONS } from '../const';
import { useI18n } from '/@/hooks/web/useI18n';

interface ItemRender {
  page: number;
  type: 'page' | 'prev' | 'next';
  originalElement: any;
}

function itemRender({ page, type, originalElement }: ItemRender) {
  if (type === 'prev') {
    return page === 0 ? null : <LeftOutlined />;
  } else if (type === 'next') {
    return page === 1 ? null : <RightOutlined />;
  }
  return originalElement;
}

export function usePagination(refProps: ComputedRef<BasicTableProps>) {
  const { t } = useI18n();

  const configRef = ref<PaginationProps>({});
  const show = ref(true);

  watch(
    () => unref(refProps).pagination,
    (pagination) => {
      if (!isBoolean(pagination) && pagination) {
        configRef.value = {
          ...unref(configRef),
          ...(pagination ?? {}),
        };
      }
    },
  );

  const getPaginationInfo = computed((): PaginationProps | boolean => {
    const { pagination } = unref(refProps);

    if (!unref(show) || (isBoolean(pagination) && !pagination)) {
      return false;
    } else if (!isBoolean(pagination) && pagination) {
      if (pagination.pageSize) {
        if (!PAGE_SIZE_OPTIONS.includes(pagination.pageSize.toString())) {
          PAGE_SIZE_OPTIONS.push(pagination.pageSize.toString());
          PAGE_SIZE_OPTIONS.sort((a: any, b: any) => a - b);
        }
      }
    }
    return {
      current: 1,
      pageSize: PAGE_SIZE,
      size: 'big',
      defaultPageSize: PAGE_SIZE,
      showTotal: (total) => t(`共 ${total} 条数据`),
      showSizeChanger: true,
      pageSizeOptions: PAGE_SIZE_OPTIONS,
      itemRender: itemRender,
      showQuickJumper: true,
      ...(isBoolean(pagination) ? {} : pagination),
      ...unref(configRef),
    };
  });

  function setPagination(info: Partial<PaginationProps>) {
    const paginationInfo = unref(getPaginationInfo);
    configRef.value = {
      ...(!isBoolean(paginationInfo) ? paginationInfo : {}),
      ...info,
    };
  }

  function getPagination() {
    return unref(getPaginationInfo);
  }

  function getShowPagination() {
    const { pagination } = unref(refProps);
    if (!unref(show) || (isBoolean(pagination) && !pagination)) {
      return false;
    }
    if (unref(configRef).hideOnSinglePage) {
      if (unref(configRef).total && unref(configRef).pageSize) {
        return unref(configRef).total! > unref(configRef).pageSize! ? true : false;
      } else {
        return false;
      }
    } else {
      if (
        unref(configRef).total ||
        (unref(configRef).current !== undefined &&
          unref(configRef).current !== 1 &&
          unref(configRef).total)
      ) {
        return true;
      } else {
        return false;
      }
    }
  }

  async function setShowPagination(flag: boolean) {
    show.value = flag;
  }

  return { getPagination, getPaginationInfo, setShowPagination, getShowPagination, setPagination };
}
