import {
  AiCreateTablePageModel,
  AiCreateTablePageParams,
  AiCreateTablePageResult,
  AiTableModal,
} from './model/AiCreateTableModel';
import { defHttp } from '/@/utils/http/axios';
import { ErrorMessageMode } from '/#/axios';

enum Api {
  Page = '/form/ai-create-table/page',
  Info = '/form/ai-create-table/info',
  AiCreateTable = '/form/ai-create-table',
  TableInfo = '/form/ai-create-table/get-ai-table-info',
  AddTable = '/form/ai-create-table/create-table',
  saveForm = '/system/generator/generator-code/ai-table',
  tableSql = '/system/generator/get-create-table-sql',
  formInfo = '/system/generator/get-ai-table-info',
  ExampleData = '/system/ai/scheduling-example-data',
  ContractReview = '/system/ai/kimi/contract-review',
  Scheduling = '/system/ai/kimi/scheduling',
  WriteDownload = '/system/ai/download-file',
  WriteContent = '/system/ai/cache-content',
}

/**
 * @description: 查询AiCreateTable分页列表
 */
export async function getAiCreateTablePage(
  params: AiCreateTablePageParams,
  mode: ErrorMessageMode = 'modal',
) {
  return defHttp.get<AiCreateTablePageResult>(
    {
      url: Api.Page,
      params,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 获取AiCreateTable信息
 */
export async function getAiCreateTable(id: String, mode: ErrorMessageMode = 'modal') {
  return defHttp.get<AiCreateTablePageModel>(
    {
      url: Api.Info,
      params: { id },
    },
    {
      errorMessageMode: mode,
    },
  );
}
/**
 * @description: 获取Table信息
 */
export async function getTableInfo(
  params: { id?: string; columName?: string; tableName: string; databaseId: string },
  mode: ErrorMessageMode = 'modal',
) {
  return defHttp.post(
    {
      url: Api.TableInfo,
      params,
      timeout: 60 * 1000,
    },
    {
      errorMessageMode: mode,
    },
  );
}
/**
 * @description: 新增Ai数据库表
 */
export async function addAiDbTable(params: AiTableModal, mode: ErrorMessageMode = 'modal') {
  return defHttp.post<boolean>(
    {
      url: Api.AddTable,
      params,
    },
    {
      errorMessageMode: mode,
    },
  );
}
/**
 * @description: 新增AiCreateTable
 */
export async function addAiCreateTable(
  aiCreateTable: Recordable,
  mode: ErrorMessageMode = 'modal',
) {
  return defHttp.post<boolean>(
    {
      url: Api.AiCreateTable,
      params: aiCreateTable,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 更新AiCreateTable
 */
export async function updateAiCreateTable(
  aiCreateTable: Recordable,
  mode: ErrorMessageMode = 'modal',
) {
  return defHttp.put<boolean>(
    {
      url: Api.AiCreateTable,
      params: aiCreateTable,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 删除AiCreateTable（批量删除）
 */
export async function deleteAiCreateTable(ids: string[], mode: ErrorMessageMode = 'modal') {
  return defHttp.delete<boolean>(
    {
      url: Api.AiCreateTable,
      data: ids,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 新增Ai数据模板
 */
export async function addAiForm(params, mode: ErrorMessageMode = 'modal') {
  return defHttp.post(
    {
      url: Api.saveForm,
      params,
    },
    {
      errorMessageMode: mode,
    },
  );
}
/**
 * @description: 根据表单结构及数据库id，获取建表语句
 */
export async function getTableSql(params, mode: ErrorMessageMode = 'modal') {
  return defHttp.post(
    {
      url: Api.tableSql,
      params,
    },
    {
      errorMessageMode: mode,
    },
  );
}
/**
 * @description: 根据id获取表单详情
 */
export async function getAIFormInfo(id, mode: ErrorMessageMode = 'modal') {
  return defHttp.get(
    {
      url: Api.formInfo,
      params: { id },
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: AI生产调度列表数据
 */
export async function getAiProData(mode: ErrorMessageMode = 'modal') {
  return defHttp.get(
    {
      url: Api.ExampleData,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: AI合同评审
 */
export async function getContractReview(fileId: string, mode: ErrorMessageMode = 'modal') {
  return defHttp.download(
    {
      url: Api.ContractReview,
      method: 'get',
      params: { fileId },
      responseType: 'blob',
      timeout: 100 * 1000,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: AI生产调度查询
 */
export async function getScheduling(params, mode: ErrorMessageMode = 'modal') {
  return defHttp.post(
    {
      url: Api.Scheduling,
      params,
      timeout: 100 * 1000,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: AI写作下载
 */
export async function downLoadAIAnswer(params, mode: ErrorMessageMode = 'modal') {
  return defHttp.download(
    {
      url: Api.WriteDownload,
      method: 'get',
      params,
      responseType: 'blob',
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: AI写作下载前调用
 */
export async function sendAIAnswer(content, mode: ErrorMessageMode = 'modal') {
  return defHttp.post<boolean>(
    {
      url: Api.WriteContent,
      params: { content },
    },
    {
      errorMessageMode: mode,
    },
  );
}
