import { defHttp } from '/@/utils/http/axios';

import { ErrorMessageMode } from '/#/axios';
import {
  UnitPageListSearchModel,
  UnitAddParamsModel,
  UnitUpdateParamsModel,
  UnitPageListResultModel,
} from './model';

enum Api {
  Page = '/caseErpUnit/caseErpUnit/page',
  List = '/caseErpUnit/caseErpUnit/list',
  Unit = '/caseErpUnit/caseErpUnit',
  Info = '/caseErpUnit/caseErpUnit/info',
  Export = '/caseErpUnit/caseErpUnit/export',
  Import = '/caseErpUnit/caseErpUnit/import',
  TypeList = '/caseErpUnit/caseErpUnit/unit-list-by-unit-type',
}

/**
 * @description: 查询单位列表（分页）
 */
export async function getUnitPageList(
  params?: UnitPageListSearchModel,
  mode: ErrorMessageMode = 'modal',
) {
  return defHttp.get<UnitPageListResultModel>(
    {
      url: Api.Page,
      params,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 查询单位列表（不分页）
 */
export async function getUnitList(
  params?: UnitPageListSearchModel,
  mode: ErrorMessageMode = 'modal',
) {
  return defHttp.get<UnitPageListResultModel>(
    {
      url: Api.List,
      params,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 新增单位
 */
export async function addUnitList(type: UnitAddParamsModel, mode: ErrorMessageMode = 'modal') {
  return defHttp.post<number>(
    {
      url: Api.Unit,
      params: type,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 修改单位
 */
export async function updateUnitList(
  type: UnitUpdateParamsModel,
  mode: ErrorMessageMode = 'modal',
) {
  return defHttp.put<number>(
    {
      url: Api.Unit,
      params: type,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 删除单位（批量删除）
 */
export async function deleteUnitList(ids: string[], mode: ErrorMessageMode = 'modal') {
  return defHttp.delete<number>(
    {
      url: Api.Unit,
      data: ids,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 查询单位详情
 */
export async function getUnitListInfo(id: string, mode: ErrorMessageMode = 'modal') {
  return defHttp.get<UnitAddParamsModel>(
    {
      url: Api.Info,
      params: { id },
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 导出
 */
export async function exportInfo(ids: string[], mode: ErrorMessageMode = 'modal') {
  return defHttp.download(
    {
      url: Api.Export,
      method: 'POST',
      params: ids,
      responseType: 'blob',
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 下载模板
 */
export async function downloadTemplate(mode: ErrorMessageMode = 'modal') {
  return defHttp.download(
    {
      url: Api.Export,
      method: 'GET',
      responseType: 'blob',
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 导入
 */
export async function importInfo(params, mode: ErrorMessageMode = 'modal') {
  return defHttp.post(
    {
      url: Api.Import,
      params,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 根据单位类型id查询所有单位信息
 */
export async function getUnitListByTypeId(unitTypeId: string, mode: ErrorMessageMode = 'modal') {
  return defHttp.get<UnitUpdateParamsModel[]>(
    {
      url: Api.TypeList,
      params: { unitTypeId },
    },
    {
      errorMessageMode: mode,
    },
  );
}
