import type { AppRouteRecordRaw, Menu, subSys } from '/@/router/types';

import { defineStore } from 'pinia';
import { store } from '/@/store';
import { useI18n } from '/@/hooks/web/useI18n';
import { useUserStore } from './user';
import { useMultipleTabStore } from './multipleTab';

import { useAppStoreWithOut } from './app';
import { toRaw } from 'vue';
import { transformObjToRoute, flatMultiLevelRoutes } from '/@/router/helper/routeHelper';
import { transformRouteToMenu } from '/@/router/helper/menuHelper';

import projectSetting from '/@/settings/projectSetting';

import { PermissionModeEnum } from '/@/enums/appEnum';

import { asyncRoutes } from '/@/router/routes';
import { ERROR_LOG_ROUTE, PAGE_NOT_FOUND_ROUTE } from '/@/router/routes/basic';

import { filter } from '/@/utils/helper/treeHelper';

import { useMessage } from '/@/hooks/web/useMessage';
import { PageEnum } from '/@/enums/pageEnum';
import { getMenuList, getPermCode } from '/@/api/system/login';
import { MenuAuthModel } from '/@/api/system/login/model';
import { getSubSystemList } from '/@/api/system/subSystem';

interface PermissionState {
  // Permission code list
  permCodeList: MenuAuthModel[];
  // Whether the route has been dynamically added
  isDynamicAddedRoute: boolean;
  // To trigger a menu update
  lastBuildMenuTime: number;
  // Backstage menu list
  backMenuList: Menu[];
  frontMenuList: Menu[];
  subSystem: string;
  subSystemList: subSys[];
}
export const usePermissionStore = defineStore({
  id: 'app-permission',
  state: (): PermissionState => ({
    permCodeList: [],
    // Whether the route has been dynamically added
    isDynamicAddedRoute: false,
    // To trigger a menu update
    lastBuildMenuTime: 0,
    // Backstage menu list
    backMenuList: [],
    // menu List
    frontMenuList: [],
    //sub system
    subSystem: '',
    subSystemList: [],
  }),
  getters: {
    getPermCodeList(): MenuAuthModel[] {
      return this.permCodeList || [];
    },
    getBackMenuList(): Menu[] {
      return this.backMenuList.filter((item) => {
        const systemId = this.getSubSystem;
        return systemId === item.meta!.systemId;
      });
    },
    getFrontMenuList(): Menu[] {
      return this.frontMenuList;
    },
    getLastBuildMenuTime(): number {
      return this.lastBuildMenuTime;
    },
    getIsDynamicAddedRoute(): boolean {
      return this.isDynamicAddedRoute;
    },
    getSubSystem(): string {
      return this.subSystem;
    },
    getSubSysList(): subSys[] {
      return this.subSystemList;
    },
  },
  actions: {
    setPermCodeList(permList: MenuAuthModel[]) {
      this.permCodeList = permList;
    },

    setBackMenuList(list: Menu[]) {
      this.backMenuList = list;
      list?.length > 0 && this.setLastBuildMenuTime();
    },

    setFrontMenuList(list: Menu[]) {
      this.frontMenuList = list;
    },

    setLastBuildMenuTime() {
      this.lastBuildMenuTime = new Date().getTime();
    },

    setDynamicAddedRoute(added: boolean) {
      this.isDynamicAddedRoute = added;
    },
    setSubSystem(systemId: string) {
      this.subSystem = systemId;
    },
    setSubSystemList(subSystemList: subSys[]) {
      this.subSystemList = subSystemList;
    },
    resetState(): void {
      this.isDynamicAddedRoute = false;
      this.permCodeList = [];
      this.backMenuList = [];
      this.lastBuildMenuTime = 0;
    },
    async changeSubsystem(getShowTopMenu, getIsMobile) {
      const arr = await getSubSystemList();
      let subSystemList: any = [];
      if (getShowTopMenu && !getIsMobile) {
        arr.forEach((o) => {
          subSystemList.push({
            text: o.name,
            event: o.id,
            icon: o.icon,
          });
        });
      } else {
        subSystemList = arr;
      }
      this.setSubSystemList(subSystemList);
      if (subSystemList.length && !this.getSubSystem) {
        this.setSubSystem(arr[0].id);
      }
    },
    async changePermissionCode() {
      const userStore = useUserStore();
      const userInfo = userStore.getUserInfo;

      const permResult = await getPermCode();
      userInfo.departmentId = permResult.departmentId;
      userInfo.departmentName = permResult.departmentName;
      userInfo.postId = permResult.postId;
      userInfo.postName = permResult.postName;
      userInfo.roleName = permResult.roleName;
      userInfo.roleId = permResult.roleId;
      userInfo.desktopSchema = permResult.desktopSchema;
      userStore.setUserInfo(userInfo);
      this.setPermCodeList(permResult.menuAuthList);
    },
    async buildRoutesAction(isRefrashPermisson = true, data?): Promise<AppRouteRecordRaw[]> {
      const { t } = useI18n();
      const userStore = useUserStore();
      const appStore = useAppStoreWithOut();
      const tabStore = useMultipleTabStore();

      let routes: AppRouteRecordRaw[] = [];
      const roleList = toRaw(userStore.getRoleList) || [];
      const { permissionMode = projectSetting.permissionMode } = appStore.getProjectConfig;

      const routeFilter = (route: AppRouteRecordRaw) => {
        const { meta } = route;
        const { roles } = meta || {};
        if (!roles) return true;
        return roleList.some((role) => roles.includes(role));
      };

      const routeRemoveIgnoreFilter = (route: AppRouteRecordRaw) => {
        const { meta } = route;
        const { ignoreRoute } = meta || {};
        return !ignoreRoute;
      };

      /**
       * @description 根据设置的首页path，修正routes中的affix标记（固定首页）
       * */
      const patchHomeAffix = (routes: AppRouteRecordRaw[]) => {
        if (!routes || routes.length === 0) return;
        let homePath: string = userStore.getUserInfo.homePath || PageEnum.BASE_HOME;
        function patcher(routes: AppRouteRecordRaw[], parentPath = '') {
          if (parentPath) parentPath = parentPath + '/';
          routes.forEach((route: AppRouteRecordRaw) => {
            const { path, children, redirect } = route;
            const currentPath = path.startsWith('/') ? path : parentPath + path;
            if (currentPath === homePath) {
              if (redirect) {
                homePath = route.redirect! as string;
              } else {
                route.meta = Object.assign({}, route.meta, { affix: true });
                throw new Error('end');
              }
            }
            children && children.length > 0 && patcher(children, currentPath);
          });
        }
        try {
          patcher(routes);
        } catch (e) {
          // 已处理完毕跳出循环
        }
        return;
      };

      switch (permissionMode) {
        case PermissionModeEnum.ROLE:
          routes = filter(asyncRoutes, routeFilter);
          routes = routes.filter(routeFilter);
          // Convert multi-level routing to level 2 routing
          routes = flatMultiLevelRoutes(routes);
          break;

        case PermissionModeEnum.ROUTE_MAPPING:
          routes = filter(asyncRoutes, routeFilter);
          routes = routes.filter(routeFilter);
          const menuList = transformRouteToMenu(routes, true);
          routes = filter(routes, routeRemoveIgnoreFilter);
          routes = routes.filter(routeRemoveIgnoreFilter);
          menuList.sort((a, b) => {
            return (a.meta?.orderNo || 0) - (b.meta?.orderNo || 0);
          });

          this.setFrontMenuList(menuList);
          // Convert multi-level routing to level 2 routing
          routes = flatMultiLevelRoutes(routes);
          break;

        //  If you are sure that you do not need to do background dynamic permissions, please comment the entire judgment below
        case PermissionModeEnum.BACK:
          const { createMessage } = useMessage();

          createMessage.loading({
            content: t('菜单加载中...'),
            duration: 1,
          });

          // !Simulate to obtain permission codes from the background,
          // this function may only need to be executed once, and the actual project can be put at the right time by itself
          let routeList: AppRouteRecordRaw[] = [];
          try {
            if (isRefrashPermisson) await this.changePermissionCode();

            routeList = (await getMenuList()) as AppRouteRecordRaw[];
            if (data) {
              const needChangeTab = tabStore.getTabList.filter((tab) => {
                return tab.meta.menuId === data.id || data.childIds.includes(tab.meta.menuId);
              });
              if (needChangeTab.length) {
                const info = [];
                changeTabsData(routeList, needChangeTab[0], info);
                tabStore.setTabList(info[0]);
              }
            }
          } catch (error) {
            console.error(error);
          }

          // Dynamically introduce components
          routeList = transformObjToRoute(routeList);

          //  Background routing to menu structure
          const backMenuList = transformRouteToMenu(routeList);

          this.setBackMenuList(backMenuList);

          // remove meta.ignoreRoute item
          routeList = filter(routeList, routeRemoveIgnoreFilter);
          routeList = routeList.filter(routeRemoveIgnoreFilter);

          routeList = flatMultiLevelRoutes(routeList);
          routes = [PAGE_NOT_FOUND_ROUTE, ...routeList];

          break;
      }

      routes.push(ERROR_LOG_ROUTE);
      patchHomeAffix(routes);
      return routes;
    },
  },
});

function changeTabsData(routes, data, info) {
  for (const key in routes) {
    if (routes[key].id === data.meta.menuId) {
      info.push(routes[key]);
      return true;
    } else {
      changeTabsData(routes[key].children, data, info);
    }
  }
}

// Need to be used outside the setup
export function usePermissionStoreWithOut() {
  return usePermissionStore(store);
}
