import { FormProps, FormSchema } from '/@/components/Form';
import { BasicColumn } from '/@/components/Table';
import { getInventoryByFilter } from '/@/api/ckgl/kcpd';
import type { InventoryFilterParams, InventoryItem } from '/@/api/ckgl/kcpd/model/KcpdModel';
import { log } from 'console';

const parseSelectValueToArray = (value: unknown): string[] => {
  if (Array.isArray(value)) return value.filter(Boolean) as string[];
  if (typeof value === 'string') {
    return value
      ? value
          .split(',')
          .map((s) => s.trim())
          .filter(Boolean)
      : [];
  }
  if (value === undefined || value === null || value === '') return [];
  return [String(value)];
};

async function handleStartInventory(_schema, formModel, formActionType) {
  if (formModel?.id) {
    formActionType?.showMessage?.('编辑状态不可开始盘点');
    return;
  }

  const payload: InventoryFilterParams = {
    // 这里的 ckwz/cp 在 schema 中已配置为保存 label(名称)
    ckwz: parseSelectValueToArray(formModel?.ckwz),
    cplx: parseSelectValueToArray(formModel?.cplx),
    cp: parseSelectValueToArray(formModel?.cp),
  };
  console.log('[库存盘点] 请求参数:', payload);

  if (!payload.ckwz.length && !payload.cplx.length && !payload.cp.length) {
    formActionType?.showMessage?.('请先选择仓库位置、产品类型或产品');
    return;
  }

  try {
    const res = await getInventoryByFilter(payload, 'message');
    console.log('[库存盘点] /ckgl/kczb/Inventory 返回:', res);
    // 注意：defHttp 可能已解包，res 可能直接就是 data 数组
    const inventoryList = Array.isArray(res) ? res : res?.data || [];
    console.log('[库存盘点] /ckgl/kczb/Inventory 解析后的列表:', inventoryList);
    const list =
      (inventoryList || []).map((item: InventoryItem) => ({
        id: item.id,
        cpbh: item.cpbh,
        cp: item.cpmc,
        cplx: item.cplx,
        wz: item.hw,
        pch: item.pch,
        pdsl: item.pdsl ?? 0,
        zbid: item.zbid,
        ywzz: item.ywzz,
        ck: item.ck,
      })) || [];
    await formActionType?.setFieldsValue?.({ mesWarehouseInventoryRelaList: list });
    formActionType?.showMessage?.(list.length ? '开始盘点成功' : '未查询到匹配数据');
  } catch (error) {
    console.error('开始盘点失败', error);
    formActionType?.showMessage?.('开始盘点失败，请稍后重试');
  }
}

async function handleProfitLoss(_schema, formModel, formActionType) {
  try {
    const inventoryDetails = formModel.mesWarehouseInventoryRelaList || [];
    if (inventoryDetails.length === 0) {
      formActionType?.showMessage?.('没有盘点明细数据可处理', { type: 'warning' });
      return;
    }
    
    await formActionType?.httpRequest({
      requestUrl: '/ckgl/kczb/handleyk',
      requestType: 'post',
      data: inventoryDetails,
      headers: { Authorization: localStorage.getItem('token') }
    });
    console.log(formActionType);
    formActionType?.showMessage?.('盈亏处理成功！', { type: 'success' });
    if (window.parent && window.parent.postMessage) {
      window.parent.postMessage({ type: 'CLOSE_MODAL' }, '*');
    }
  } catch (error) {
    console.error('盈亏处理失败', error);
    formActionType?.showMessage?.('盈亏处理失败，请重试！');
  }
}

async function handleCancelInventory(_schema, formModel, formActionType) {
  try {
    // Update inventory status to "取消"
    formModel.zt = '取消';
    
    formActionType?.showMessage?.('盘点已取消', { type: 'success' });
    
    // Trigger the confirm button click event to close the modal and save changes
    if (window.parent && window.parent.postMessage) {
      window.parent.postMessage({ type: 'TRIGGER_CONFIRM_BUTTON' }, '*');
    }
  } catch (error) {
    console.error('取消盘点失败', error);
    formActionType?.showMessage?.('取消盘点失败，请重试！');
  }
}

export const searchFormSchema: FormSchema[] = [
  {
    field: 'pdrq',
    label: '盘点日期',
    defaultValue: undefined,
    component: 'RangePicker',
    componentProps: {
      format: 'YYYY-MM-DD HH:mm:ss',
      style: { width: '100%' },
      getPopupContainer: () => document.body,
    },
  },

  {
    field: 'pdbh',
    label: '盘点编码',
    defaultValue: undefined,
    component: 'Input',
  },
  {
    field: 'zt',
    label: '盘点状态',
    defaultValue: undefined,
    component: 'XjrSelect',
    componentProps: {
      datasourceType: 'dic',
      params: { itemId: '2018581306872238081' },
      labelField: 'name',
      valueField: 'value',
      mode: 'multiple',
      showSearch: true,
      getPopupContainer: () => document.body,
    },
  },
];

export const columns: BasicColumn[] = [
  {
    resizable: true,
    dataIndex: 'pdbh',
    title: '盘点编码',
    componentType: 'auto-code',

    fixed: false,
    sorter: true,

    styleConfig: undefined,
    listStyle: undefined,
  },

  {
    resizable: true,
    dataIndex: 'pdrq',
    title: '盘点日期',
    componentType: 'date',

    fixed: false,
    sorter: true,

    styleConfig: undefined,
    listStyle: undefined,
  },

  {
    resizable: true,
    dataIndex: 'zt',
    title: '盘点状态',
    componentType: 'select',

    fixed: false,
    sorter: true,

    styleConfig: undefined,
    listStyle: undefined,
  },

  {
    resizable: true,
    dataIndex: 'bz',
    title: '备注',
    componentType: 'input',

    fixed: false,
    sorter: true,

    styleConfig: undefined,
    listStyle: '',
  },
];
//表头合并配置
export const headerMergingData = [];
//表单事件
export const formEventConfigs = {
  0: [
    {
      type: 'circle',
      color: '#2774ff',
      text: '开始节点',
      icon: '#icon-kaishi',
      bgcColor: '#D8E5FF',
      isUserDefined: false,
    },
    {
      color: '#F6AB01',
      icon: '#icon-chushihua',
      text: '初始化表单',
      bgcColor: '#f9f5ea',
      isUserDefined: false,
      nodeInfo: { processEvent: [] },
    },
  ],
  1: [
    {
      color: '#B36EDB',
      icon: '#icon-shujufenxi',
      text: '获取表单数据',
      detail: '(新增无此操作)',
      bgcColor: '#F8F2FC',
      isUserDefined: false,
      nodeInfo: { processEvent: [] },
    },
  ],
  2: [
    {
      color: '#F8625C',
      icon: '#icon-jiazai',
      text: '加载表单',
      bgcColor: '#FFF1F1',
      isUserDefined: false,
      nodeInfo: { processEvent: [] },
    },
  ],
  3: [
    {
      color: '#6C6AE0',
      icon: '#icon-jsontijiao',
      text: '提交表单',
      bgcColor: '#F5F4FF',
      isUserDefined: false,
      nodeInfo: { processEvent: [] },
    },
  ],
  4: [
    {
      type: 'circle',
      color: '#F8625C',
      text: '结束节点',
      icon: '#icon-jieshuzhiliao',
      bgcColor: '#FFD6D6',
      isLast: true,
      isUserDefined: false,
    },
  ],
};
export const formProps: FormProps = {
  labelCol: { span: 3, offset: 0 },
  labelAlign: 'right',
  layout: 'horizontal',
  size: 'default',
  buttonList: [],
  schemas: [
    {
      key: '44f1b3495b854c708ce9f13c0e6afc94',
      field: '',
      label: '',
      type: 'grid',
      colProps: { span: 24 },
      component: 'Grid',
      children: [
        {
          span: 2,
          list: [
            {
              key: '3498b1519f97476e87627a4b7de4b268',
              label: '开始盘点',
              type: 'button',
              component: 'Button',
              colProps: { span: 24 },
              componentProps: {
                name: '开始盘点',
                buttonStyleType: 'primary',
                buttonWidth: '',
                buttonHeight: '',
                datasourceType: 'api',
                prefix: '',
                suffix: '',
                labelField: 'label',
                valueField: 'value',
                apiConfig: {
                  path: 'CodeGeneration/assoc-selection',
                  method: 'GET',
                  apiId: 'ac61f579074c465aaf017d64284a44a3',
                  outputParams: [
                    { name: 'label', tableTitle: 'label' },
                    { name: 'value', tableTitle: 'value' },
                  ],
                },
                dicOptions: [],
                disabled: true,
                isShow: true,
                margin: '10px',
                events: { click: handleStartInventory },
                type: 1,
                event: [],
                tooltipConfig: { visible: false, title: '提示文本' },
                span: 7,
                width: '',
                height: '',
              },
            },
          ],
        },
        {
          span: 2,
          list: [
            {
              key: 'b0b8fc72994d4f34a985bbf6880799f1',
              label: '确认盘点',
              type: 'button',
              component: 'Button',
              colProps: { span: 24 },
              componentProps: {
                name: '确认盘点',
                buttonStyleType: 'primary',
                buttonWidth: '',
                buttonHeight: '',
                datasourceType: 'api',
                prefix: '',
                suffix: '',
                labelField: 'label',
                valueField: 'value',
                apiConfig: {
                  path: 'CodeGeneration/assoc-selection',
                  method: 'GET',
                  apiId: 'ac61f579074c465aaf017d64284a44a3',
                  outputParams: [
                    { name: 'label', tableTitle: 'label' },
                    { name: 'value', tableTitle: 'value' },
                  ],
                },
                dicOptions: [],
                disabled: true,
                isShow: true,
                margin: '10px',
                events: { click: "formModel.zt = '已完成'; formActionType.showMessage('盘点已确认，状态已更新为已完成', { type: 'success' });" },
                type: 1,
                event: [],
                tooltipConfig: { visible: false, title: '提示文本' },
                span: 7,
                width: '',
                height: '',
              },
            },
          ],
        },
        {
          span: 2,
          list: [
            {
              key: 'a2539c9a104248ab9cb0dedea1641b39',
              label: '取消盘点',
              type: 'button',
              component: 'Button',
              colProps: { span: 24 },
              componentProps: {
                name: '取消盘点',
                buttonStyleType: 'primary',
                buttonWidth: '',
                buttonHeight: '',
                datasourceType: 'api',
                prefix: '',
                suffix: '',
                labelField: 'label',
                valueField: 'value',
                apiConfig: {
                  path: 'CodeGeneration/assoc-selection',
                  method: 'GET',
                  apiId: 'ac61f579074c465aaf017d64284a44a3',
                  outputParams: [
                    { name: 'label', tableTitle: 'label' },
                    { name: 'value', tableTitle: 'value' },
                  ],
                },
                dicOptions: [],
                disabled: false,
                isShow: true,
                margin: '10px',
                events: { click: handleCancelInventory },
                type: 1,
                event: [],
                tooltipConfig: { visible: false, title: '提示文本' },
                span: 7,
                width: '',
                height: '',
              },
            },
          ],
        },
        {
          span: 2,
          list: [
            {
              key: 'f0dcaa5c8080475f94114c779b55bd7f',
              label: '处理盈亏',
              type: 'button',
              component: 'Button',
              colProps: { span: 24 },
              componentProps: {
                name: '处理盈亏',
                buttonStyleType: 'primary',
                buttonWidth: '',
                buttonHeight: '',
                datasourceType: 'api',
                prefix: '',
                suffix: '',
                labelField: 'label',
                valueField: 'value',
                apiConfig: {
                  path: 'CodeGeneration/assoc-selection',
                  method: 'GET',
                  apiId: 'ac61f579074c465aaf017d64284a44a3',
                  outputParams: [
                    { name: 'label', tableTitle: 'label' },
                    { name: 'value', tableTitle: 'value' },
                  ],
                },
                dicOptions: [],
                disabled: false,
                isShow: true,
                margin: '10px',
                events: { click: handleProfitLoss },
                modal: null,
                type: 1,
                event: [],
                tooltipConfig: { visible: false, title: '提示文本' },
                span: 7,
                width: '',
                height: '',
              },
            },
          ],
        },
      ],
      componentProps: {
        gutter: 0,
        justify: 'start',
        align: 'top',
        isShow: true,
        showBorder: false,
        bordercolor: '#d9d9d9',
        bordershowtype: [true, true, true, true],
        borderwidth: 1,
        padding: '10px',
        margin: '10px',
      },
    },
    {
      key: 'bd276037af164568843b957a4de98cb2',
      field: '',
      label: '',
      type: 'grid',
      colProps: { span: 24 },
      component: 'Grid',
      children: [
        {
          span: 8,
          list: [
            {
              key: 'abf0de1b03ff413181086dd3c78c1333',
              field: 'ckwz',
              label: '仓库位置',
              type: 'select',
              component: 'XjrSelect',
              colProps: { span: 24 },
              componentProps: {
                width: '100%',
                span: 4,
                placeholder: '请选择仓库位置',
                showLabel: true,
                showSearch: true,
                isMultiple: true,
                clearable: false,
                disabled: true,
                staticOptions: [
                  { key: 1, label: 'Option 1', value: 'Option 1' },
                  { key: 2, label: 'Option 2', value: 'Option 2' },
                  { key: 3, label: 'Option 3', value: 'Option 3' },
                ],
                datasourceType: 'api',
                labelField: 'label',
                // 保存名称(=label)，开始盘点接口要求传名称
                valueField: 'label',
                apiConfig: {
                  path: '/getStoreLocation',
                  method: 'GET',
                  apiId: 'd55f1866db284ecd8b038a3956b9c5b7',
                  apiParams: [
                    { key: '1', title: 'Query Params', tableInfo: [] },
                    { key: '2', title: 'Header', tableInfo: [] },
                    { key: '3', title: 'Body' },
                  ],
                  script:
                    'var sql="select id as value,name as label from mes_storage_location where delete_mark = 0 ";\r\nreturn db.select(sql);',
                },
                dicOptions: [],
                required: true,
                rules: [],
                events: {},
                isShow: true,
                tooltipConfig: { visible: false, title: '提示文本' },
                itemId: '2010539474382962690',
                style: { width: '100%' },
              },
            },
          ],
        },
        {
          span: 8,
          list: [
            {
              key: '4d9b8b91d29b4c54b7fcf13ce3f0d44d',
              field: 'cplx',
              label: '产品类型',
              type: 'select',
              component: 'XjrSelect',
              colProps: { span: 24 },
              componentProps: {
                width: '100%',
                span: 4,
                placeholder: '请选择产品类型类型',
                showLabel: true,
                showSearch: true,
                isMultiple: true,
                clearable: false,
                disabled: true,
                staticOptions: [
                  { key: 1, label: 'Option 1', value: 'Option 1' },
                  { key: 2, label: 'Option 2', value: 'Option 2' },
                  { key: 3, label: 'Option 3', value: 'Option 3' },
                ],
                datasourceType: 'dic',
                params: { itemId: '2003761375221407745' },
                labelField: 'name',
                valueField: 'value',
                apiConfig: {
                  path: 'CodeGeneration/selection',
                  method: 'GET',
                  apiId: '93d735dcb7364a0f8102188ec4d77ac7',
                },
                dicOptions: [],
                required: false,
                rules: [],
                events: {},
                isShow: true,
                tooltipConfig: { visible: false, title: '提示文本' },
                itemId: '2003761375221407745',
                style: { width: '100%' },
              },
            },
          ],
        },
        {
          span: 8,
          list: [
            {
              key: '0e486558cc20447db6afbd4e24d22140',
              field: 'cp',
              label: '产品',
              type: 'select',
              component: 'XjrSelect',
              colProps: { span: 24 },
              componentProps: {
                width: '100%',
                span: 4,
                placeholder: '请选择产品',
                showLabel: true,
                showSearch: true,
                isMultiple: true,
                clearable: false,
                disabled: true,
                staticOptions: [
                  { key: 1, label: 'Option 1', value: 'Option 1' },
                  { key: 2, label: 'Option 2', value: 'Option 2' },
                  { key: 3, label: 'Option 3', value: 'Option 3' },
                ],
                datasourceType: 'api',
                labelField: 'label',
                // 保存名称(=label)，开始盘点接口要求传名称
                valueField: 'label',
                apiConfig: {
                  path: '/scgl/scjh/getAllProduct',
                  method: 'GET',
                  apiId: 'f4fbb57f2f18425e97918a031c8aa7d8',
                  apiParams: [
                    { key: '1', title: 'Query Params', tableInfo: [] },
                    { key: '2', title: 'Header', tableInfo: [] },
                    { key: '3', title: 'Body' },
                  ],
                  script:
                    "var sql = 'select *,id as value,cpmc as label from mes_base_product_info where delete_mark=0';\r\nreturn db.select(sql)",
                },
                dicOptions: [],
                required: false,
                rules: [],
                events: {},
                isShow: true,
                tooltipConfig: { visible: false, title: '提示文本' },
                itemId: '2003753040338247682',
                style: { width: '100%' },
              },
            },
          ],
        },
        {
          span: 8,
          list: [
            {
              key: '6be9efdbb1b7492f943ec14cb58e2d39',
              field: 'pdrq',
              label: '盘点日期',
              type: 'date',
              component: 'DatePicker',
              colProps: { span: 24 },
              defaultValue: '',
              componentProps: {
                span: 4,
                defaultValue: '',
                width: '100%',
                placeholder: '请选择盘点日期',
                format: 'YYYY-MM-DD HH:mm:ss',
                showLabel: true,
                allowClear: true,
                disabled: true,
                required: false,
                isShow: true,
                rules: [],
                events: {},
                isGetCurrent: false,
                tooltipConfig: { visible: false, title: '提示文本' },
                searchType: 'time',
                style: { width: '100%' },
              },
            },
          ],
        },
        {
          span: 16,
          list: [
            {
              key: '6d3b646b25354a8aa1d133eb867a3331',
              field: 'bz',
              label: '备注',
              type: 'input',
              component: 'Input',
              colProps: { span: 24 },
              defaultValue: '',
              componentProps: {
                width: '100%',
                span: 2,
                defaultValue: '',
                placeholder: '请输入备注',
                prefix: '',
                suffix: '',
                addonBefore: '',
                addonAfter: '',
                disabled: true,
                allowClear: false,
                showLabel: true,
                required: false,
                rules: [],
                events: {},
                listStyle: '',
                isSave: false,
                isShow: true,
                scan: false,
                bordered: true,
                isShowAi: false,
                tooltipConfig: { visible: false, title: '提示文本' },
                style: { width: '100%' },
              },
            },
          ],
        },
        {
          span: 8,
          list: [
            {
              key: '8a6000a1cd5d485787f7dc14308dd70f',
              field: 'pdbh',
              label: '盘点编码',
              type: 'auto-code',
              component: 'AutoCodeRule',
              colProps: { span: 24 },
              componentProps: {
                width: '100%',
                span: 7,
                placeholder: '请输入盘点编码',
                prefix: '',
                suffix: '',
                addonBefore: '',
                addonAfter: '',
                showLabel: true,
                autoCodeRule: 'kcpd',
                required: false,
                isShow: false,
                tooltipConfig: { visible: false, title: '提示文本' },
                style: { width: '100%' },
              },
            },
          ],
        },
        {
          span: 8,
          list: [
            {
              key: '7feeb7d43eda46a3919bcf32e64948b6',
              field: 'zt',
              label: '盘点状态',
              type: 'select',
              component: 'XjrSelect',
              colProps: { span: 24 },
              componentProps: {
                width: '100%',
                span: 7,
                placeholder: '请选择盘点状态',
                showLabel: true,
                showSearch: false,
                isMultiple: false,
                clearable: false,
                disabled: false,
                staticOptions: [
                  { key: 1, label: 'Option 1', value: 'Option 1' },
                  { key: 2, label: 'Option 2', value: 'Option 2' },
                  { key: 3, label: 'Option 3', value: 'Option 3' },
                ],
                defaultSelect: null,
                datasourceType: 'dic',
                params: { itemId: '2018581306872238081' },
                labelField: 'name',
                valueField: 'value',
                apiConfig: {
                  path: 'CodeGeneration/selection',
                  method: 'GET',
                  apiId: '93d735dcb7364a0f8102188ec4d77ac7',
                },
                dicOptions: [],
                required: false,
                rules: [],
                events: {},
                isShow: false,
                tooltipConfig: { visible: false, title: '提示文本' },
                itemId: '2018581306872238081',
                style: { width: '100%' },
              },
            },
          ],
        },
      ],
      componentProps: {
        gutter: 16,
        justify: 'start',
        align: 'top',
        isShow: true,
        showBorder: false,
        bordercolor: '#d9d9d9',
        bordershowtype: [true, true, true, true],
        borderwidth: 1,
        padding: '10px',
        margin: '10px',
      },
    },
    {
      key: '1c89abd7a60a4f6eb53f65e53e19ebf8',
      field: '',
      label: '',
      type: 'tab',
      colProps: { span: 24 },
      component: 'Tab',
      children: [
        {
          span: 24,
          name: '盘点明细',
          prefix: '',
          suffix: '',
          activeColor: '#1c8dff',
          folderId: '',
          imageUrl: '',
          conFolderId: '',
          conImageUrl: '',
          list: [
            {
              key: 'dcd23b48f99b4954a392674fd7f967f9',
              label: '',
              field: 'mesWarehouseInventoryRelaList',
              type: 'form',
              component: 'SubForm',
              required: true,
              colProps: { span: 24 },
              componentProps: {
                mainKey: 'mesWarehouseInventoryRelaList',
                columns: [
                  {
                    key: 'a64dec69ac534215b055bd63e16d0f77',
                    title: '业务组织',
                    dataIndex: 'ywzz',
                    componentType: 'Input',
                    defaultValue: '',
                    componentProps: {
                      width: '100%',
                      span: '',
                      defaultValue: '',
                      placeholder: '请输入业务组织',
                      maxlength: null,
                      prefix: '',
                      suffix: '',
                      addonBefore: '',
                      addonAfter: '',
                      disabled: true,
                      allowClear: false,
                      showLabel: true,
                      required: false,
                      rules: [],
                      events: {},
                      listStyle: '',
                      isSave: false,
                      isShow: true,
                      scan: false,
                      bordered: true,
                      isShowAi: false,
                      tooltipConfig: { visible: false, title: '提示文本' },
                    },
                  },
                  {
                    key: '6baef041c82842dc9adb2d4cfb7b5ad3',
                    title: '产品编号',
                    dataIndex: 'cpbh',
                    componentType: 'Input',
                    defaultValue: '',
                    componentProps: {
                      width: '100%',
                      span: '',
                      defaultValue: '',
                      placeholder: '请输入产品编号',
                      maxlength: null,
                      prefix: '',
                      suffix: '',
                      addonBefore: '',
                      addonAfter: '',
                      disabled: true,
                      allowClear: false,
                      showLabel: true,
                      required: false,
                      rules: [],
                      events: {},
                      listStyle: '',
                      isSave: false,
                      isShow: false,
                      scan: false,
                      bordered: true,
                      isShowAi: false,
                      tooltipConfig: { visible: false, title: '提示文本' },
                    },
                  },
                  {
                    key: '63c46f68444c4092aff3d711a0e4f1c4',
                    title: '产品',
                    dataIndex: 'cp',
                    componentType: 'XjrSelect',
                    componentProps: {
                      width: '100%',
                      span: '',
                      placeholder: '请选择产品',
                      showLabel: true,
                      showSearch: false,
                      isMultiple: false,
                      clearable: false,
                      disabled: true,
                      staticOptions: [
                        { key: 1, label: 'Option 1', value: 'Option 1' },
                        { key: 2, label: 'Option 2', value: 'Option 2' },
                        { key: 3, label: 'Option 3', value: 'Option 3' },
                      ],
                      datasourceType: 'api',
                      labelField: 'label',
                      valueField: 'value',
                      apiConfig: {
                        path: '/scgl/scjh/getAllProduct',
                        method: 'GET',
                        apiId: 'f4fbb57f2f18425e97918a031c8aa7d8',
                        apiParams: [
                          { key: '1', title: 'Query Params', tableInfo: [] },
                          { key: '2', title: 'Header', tableInfo: [] },
                          { key: '3', title: 'Body' },
                        ],
                        script:
                          "var sql = 'select *,id as value,cpmc as label from mes_base_product_info where delete_mark=0';\r\nreturn db.select(sql)",
                      },
                      dicOptions: [],
                      required: false,
                      rules: [],
                      events: {},
                      isShow: true,
                      tooltipConfig: { visible: false, title: '提示文本' },
                      listStyle: "return 'border: 0'",
                    },
                  },
                  {
                    key: 'b52ebd44f5b1405c9d462c30c214ed6f',
                    title: '产品类型',
                    dataIndex: 'cplx',
                    componentType: 'XjrSelect',
                    componentProps: {
                      width: '100%',
                      span: '',
                      placeholder: '请选择产品类型',
                      showLabel: true,
                      showSearch: false,
                      isMultiple: false,
                      clearable: false,
                      disabled: true,
                      staticOptions: [
                        { key: 1, label: 'Option 1', value: 'Option 1' },
                        { key: 2, label: 'Option 2', value: 'Option 2' },
                        { key: 3, label: 'Option 3', value: 'Option 3' },
                      ],
                      datasourceType: 'dic',
                      params: { itemId: '2003761375221407745' },
                      labelField: 'name',
                      valueField: 'value',
                      apiConfig: {
                        path: 'CodeGeneration/selection',
                        method: 'GET',
                        apiId: '93d735dcb7364a0f8102188ec4d77ac7',
                      },
                      dicOptions: [],
                      required: false,
                      rules: [],
                      events: {},
                      isShow: true,
                      tooltipConfig: { visible: false, title: '提示文本' },
                      itemId: '2003761375221407745',
                      listStyle: "return 'border: 0'",
                    },
                  },
                  {
                    key: '14b12cc2cf3a40bdae1fa6869902252a',
                    title: '仓库',
                    dataIndex: 'ck',
                    componentType: 'Input',
                    defaultValue: '',
                    componentProps: {
                      width: '100%',
                      span: '',
                      defaultValue: '',
                      placeholder: '请输入仓库',
                      maxlength: null,
                      prefix: '',
                      suffix: '',
                      addonBefore: '',
                      addonAfter: '',
                      disabled: true,
                      allowClear: false,
                      showLabel: true,
                      required: false,
                      rules: [],
                      events: {},
                      listStyle: '',
                      isSave: false,
                      isShow: true,
                      scan: false,
                      bordered: true,
                      isShowAi: false,
                      tooltipConfig: { visible: false, title: '提示文本' },
                    },
                  },
                  {
                    key: '4e325b4ad3cf470d92af7e0d090e0a6d',
                    title: '位置',
                    dataIndex: 'wz',
                    componentType: 'XjrSelect',
                    componentProps: {
                      width: '100%',
                      span: '',
                      placeholder: '请选择位置',
                      showLabel: true,
                      showSearch: false,
                      isMultiple: false,
                      clearable: false,
                      disabled: true,
                      staticOptions: [
                        { key: 1, label: 'Option 1', value: 'Option 1' },
                        { key: 2, label: 'Option 2', value: 'Option 2' },
                        { key: 3, label: 'Option 3', value: 'Option 3' },
                      ],
                      datasourceType: 'api',
                      labelField: 'label',
                      valueField: 'value',
                      apiConfig: {
                        path: '/getStoreLocation',
                        method: 'GET',
                        apiId: 'd55f1866db284ecd8b038a3956b9c5b7',
                        apiParams: [
                          { key: '1', title: 'Query Params', tableInfo: [] },
                          { key: '2', title: 'Header', tableInfo: [] },
                          { key: '3', title: 'Body' },
                        ],
                        script:
                          'var sql="select id as value,name as label from mes_storage_location where delete_mark = 0 ";\r\nreturn db.select(sql);',
                      },
                      dicOptions: [],
                      required: false,
                      rules: [],
                      events: {},
                      isShow: true,
                      tooltipConfig: { visible: false, title: '提示文本' },
                      itemId: '2010539474382962690',
                      listStyle: "return 'border: 0'",
                    },
                  },
                  {
                    key: '37437506e48e4e63afa6b31af571b310',
                    title: '单件\\批次号',
                    dataIndex: 'pch',
                    componentType: 'Input',
                    defaultValue: '',
                    componentProps: {
                      width: '100%',
                      span: '',
                      defaultValue: '',
                      placeholder: '请输入单件\\批次号',
                      prefix: '',
                      suffix: '',
                      addonBefore: '',
                      addonAfter: '',
                      disabled: true,
                      allowClear: false,
                      showLabel: true,
                      required: false,
                      rules: [],
                      events: {},
                      listStyle: '',
                      isSave: false,
                      isShow: true,
                      scan: false,
                      bordered: true,
                      isShowAi: false,
                      tooltipConfig: { visible: false, title: '提示文本' },
                    },
                  },
                  {
                    key: 'a2b46aaba5264ee79e83eeeda69990d0',
                    title: '盘点数量',
                    dataIndex: 'pdsl',
                    componentType: 'InputNumber',
                    defaultValue: 0,
                    componentProps: {
                      width: '100%',
                      span: '',
                      defaultValue: 0,
                      min: 0,
                      step: 1,
                      disabled: false,
                      showLabel: true,
                      controls: true,
                      required: false,
                      subTotal: false,
                      isShow: true,
                      rules: [],
                      events: {},
                      tooltipConfig: { visible: false, title: '提示文本' },
                      listStyle: "return 'width:100%'",
                    },
                  },
                  { title: '操作', key: 'action', fixed: 'right', width: '50px' },
                ],
                span: '24',
                preloadType: 'api',
                apiConfig: {},
                itemId: '',
                dicOptions: [],
                useSelectButton: false,
                buttonName: '选择数据',
                showLabel: true,
                showComponentBorder: true,
                showBorder: false,
                bordercolor: '#f0f0f0',
                bordershowtype: [true, true, true, true],
                borderwidth: 1,
                showIndex: true,
                isShow: true,
                multipleHeads: [],
                buttonList: [],
                topButtonList: [],
                isExport: false,
                isImport: false,
                isDeleteSelected: false,
                isListView: false,
                viewList: [],
                isShowAdd: false,
                isShowDelete: true,
                hasCheckedCol: false,
                events: {},
                showPagenation: true,
              },
            },
          ],
        },
      ],
      componentProps: { tabPosition: 'top', size: 'default', type: 'card', isShow: true },
    },
  ],
  showActionButtonGroup: false,
  buttonLocation: 'center',
  actionColOptions: { span: 24 },
  showResetButton: false,
  showSubmitButton: false,
  hiddenComponent: [],
};

export const formButtons = [
  {
    key: 'confirm',
    code: 'confirm',
    name: '确定',
    style: 'primary',
    event: [],
    isShow: true,
    index: 2,
    type: 1,
  },
  {
    key: 'cancel',
    code: 'cancel',
    name: '取消',
    style: 'default',
    event: [],
    isShow: true,
    index: 1,
    type: 1,
  },
  {
    key: 'reset',
    code: 'reset',
    name: '重置',
    style: 'default',
    event: [],
    isShow: true,
    index: 0,
    type: 1,
  },
];
