import { getNodeName } from '/@bpmn/config/property';
import {
  BpmnNodeKey,
  CallActivityType,
  MultipleInstancesType,
  MemberType,
  ParamType,
  FinishType,
} from '/@/enums/workflowEnum';
import { formInitConfig } from '/@/model/workflow/formSetting';
import {
  ChildNodeConfig,
  ChildNodeItem,
  MenuConfig,
  ProcessConfig,
  ProcessConfigParameter,
  ProcessJsonModel,
  ValidateProcessConfigs,
} from '/@/model/workflow/workflowConfig';
import { AssignmentConfig, ParamAssignmentConfig } from '/@/model/workflow/ParamSetting';
import { useI18n } from '/@/hooks/web/useI18n';
const { t } = useI18n();
export function validateJson(processJson: ProcessJsonModel) {
  const validateProcess: ValidateProcessConfigs = [];
  validateProcess.push(...validateProcessConfig(processJson.processConfig));
  validateProcess.push(...validateChildNodeConfigJson(processJson.childNodeConfig));
  return validateProcess;
}
// 校验模板信息
function validateProcessConfig(processConfig: ProcessConfig): ValidateProcessConfigs {
  const validateProcess: ValidateProcessConfigs = [];
  if (processConfig.code == '') {
    validateProcess.push({
      nodeName: t('流程属性'),
      nodeType: BpmnNodeKey.PROCESS,
      msg: t('模板编号不能为空'),
      validate: false,
    });
  }
  if (processConfig.name == '') {
    validateProcess.push({
      nodeName: t('流程属性'),
      nodeType: BpmnNodeKey.PROCESS,
      msg: t('模板名称不能为空'),
      validate: false,
    });
  }
  if (processConfig.category == '') {
    validateProcess.push({
      nodeName: t('流程属性'),
      nodeType: BpmnNodeKey.PROCESS,
      msg: t('模板分类不能为空'),
      validate: false,
    });
  }
  validateProcess.push(...validateMenuConfig(processConfig.menuConfig));
  validateProcess.push(...validateformInitConfig(processConfig.formInitConfig));
  validateProcess.push(...validateProcessParameter(processConfig.processParamConfigs));
  return validateProcess;
}
// 菜单设置
function validateMenuConfig(menuConfig: MenuConfig): ValidateProcessConfigs {
  const validateProcess: ValidateProcessConfigs = [];
  if (menuConfig.enabled) {
    if (menuConfig.code == '') {
      validateProcess.push({
        nodeName: t('流程属性'),
        nodeType: BpmnNodeKey.PROCESS,
        msg: t('菜单设置中功能编号未填写，请确认'),
        validate: false,
      });
    }
    if (menuConfig.name == '') {
      validateProcess.push({
        nodeName: t('流程属性'),
        nodeType: BpmnNodeKey.PROCESS,
        msg: t('菜单设置中功能名称未填写，请确认'),
        validate: false,
      });
    }
  }
  return validateProcess;
}
//表单发起流程
function validateformInitConfig(formInitConfig: formInitConfig): ValidateProcessConfigs {
  const validateProcess: ValidateProcessConfigs = [];
  if (formInitConfig.enabled) {
    if (formInitConfig.formId == '') {
      validateProcess.push({
        nodeName: t('流程属性'),
        nodeType: BpmnNodeKey.PROCESS,
        msg: t('启用了表单发起流程功能，但未绑定表单，请确认'),
        validate: false,
      });
    }
  }
  return validateProcess;
}
//流程参数
function validateProcessParameter(parameter: ProcessConfigParameter): ValidateProcessConfigs {
  const validateProcess: ValidateProcessConfigs = [];
  if (Array.isArray(parameter)) {
    parameter.forEach((ele) => {
      if (ele.name == '') {
        validateProcess.push({
          nodeName: t('流程属性'),
          nodeType: BpmnNodeKey.PROCESS,
          msg: t('流程参数中有参数名称未填写，请确认'),
          validate: false,
        });
      }
    });
  }
  return validateProcess;
}
// 校验节点
function validateChildNodeConfigJson(nodes: ChildNodeConfig) {
  const validateProcess: ValidateProcessConfigs = childNodes(nodes);
  return validateProcess;
}

function childNodes(childNodes: ChildNodeConfig): ValidateProcessConfigs {
  const validateProcess: ValidateProcessConfigs = [];
  childNodes.forEach((element: ChildNodeItem) => {
    validateProcess.push(...basicNode(element));
    if (element.type == BpmnNodeKey.START) {
      validateProcess.push(...paramsNode(element));
    } else if (element.type == BpmnNodeKey.USER) {
      validateProcess.push(...userNode(element));
      validateProcess.push(...paramsNode(element));
    } else if (element.type == BpmnNodeKey.SCRIPT) {
      validateProcess.push(...scriptNode(element));
      validateProcess.push(...paramsNode(element));
    } else if (element.type == BpmnNodeKey.CALLACTIVITY) {
      validateProcess.push(...callactivityNode(element));
    }
  });
  return validateProcess;
}
function userNode(element: ChildNodeItem): ValidateProcessConfigs {
  const validateProcess: ValidateProcessConfigs = [];
  const nodeName = element.name ? element.name : getNodeName(element.type);
  // 会签
  if (element.hasOwnProperty('countersignConfig')) {
    if (element['countersignConfig'].multipleInstancesType != MultipleInstancesType.NONE) {
      if (element['countersignConfig'].countersignList.length > 0) {
        const checkedIds = element['countersignConfig'].countersignList.filter((ele) => {
          return ele.checked;
        });
        if (checkedIds.length == 0) {
          validateProcess.push({
            nodeName,
            nodeType: element.type,
            msg: '【' + nodeName + '】' + t('中会签配置中未选中任何会签人员，请确认'),
            validate: false,
          });
        }
      } else {
        validateProcess.push({
          nodeName,
          nodeType: element.type,
          msg: '【' + nodeName + '】' + t('中会签配置任何审批人员，请确认'),
          validate: false,
        });
      }
    }
  }
  //意见簿
  if (element.hasOwnProperty('opinionConfig') && element['opinionConfig'].enabled) {
    if (element['opinionConfig'].component.length == 0) {
      validateProcess.push({
        nodeName,
        nodeType: element.type,
        msg: '【' + nodeName + '】' + t('中关联意见框未配置相关意见框组件，请确认'),
        validate: false,
      });
    }
  }
  return validateProcess;
}
function scriptNode(element: ChildNodeItem): ValidateProcessConfigs {
  const validateProcess: ValidateProcessConfigs = [];
  const nodeName = element.name ? element.name : getNodeName(element.type);
  //脚本参数配置
  if (
    element.hasOwnProperty('script') &&
    element['script'].enabled &&
    element['script'].scriptContent == ''
  ) {
    validateProcess.push({
      nodeName,
      nodeType: element.type,
      msg: '【' + nodeName + '】' + t('中启用了脚本参数配置，但是未填写执行脚本，请确认'),
      validate: false,
    });
  }
  //API配置
  if (element.hasOwnProperty('api') && element['api'].enabled && element['api'].id == '') {
    validateProcess.push({
      nodeName,
      nodeType: element.type,
      msg: '【' + nodeName + '】' + t('中启用了API配置，但是未填写选中API，请确认'),
      validate: false,
    });
  }
  return validateProcess;
}
function callactivityNode(element: ChildNodeItem): ValidateProcessConfigs {
  const validateProcess: ValidateProcessConfigs = [];
  const nodeName = element.name ? element.name : getNodeName(element.type);
  if (element.hasOwnProperty('schemaId') && element['schemaId'] == '') {
    validateProcess.push({
      nodeName,
      nodeType: element.type,
      msg: '【' + nodeName + '】' + t('中启用了外部流程，但是未关联任何外部流程，请确认'),
      validate: false,
    });
  }
  if (element['inParams'] && element['inParams'].length > 0) {
    element['inParams'].forEach((item) => {
      if (item.source == '') {
        validateProcess.push({
          nodeName,
          nodeType: element.type,
          msg: '【' + nodeName + '】' + t('中输入参数中有变量来源未配置，请确认'),
          validate: false,
        });
      }
      if (item.target == '') {
        validateProcess.push({
          nodeName,
          nodeType: element.type,
          msg: '【' + nodeName + '】' + t('中输入参数中有目标变量未配置，请确认'),
          validate: false,
        });
      }
    });
  }
  // 单实例
  if (
    element.hasOwnProperty('callActivityType') &&
    element['callActivityType'] == CallActivityType.SINGLE
  ) {
    if (element['originatorNode'] == '') {
      validateProcess.push({
        nodeName,
        nodeType: element.type,
        msg: '【' + nodeName + '】' + t('中启用了外部流程，但是未配置发起人，请确认'),
        validate: false,
      });
    }
    if (element['outParams'] && element['outParams'].length > 0) {
      element['outParams'].forEach((item) => {
        if (item.source == '') {
          validateProcess.push({
            nodeName,
            nodeType: element.type,
            msg: '【' + nodeName + '】' + t('中输入参数中有变量来源未配置，请确认'),
            validate: false,
          });
        }
        if (item.target == '') {
          validateProcess.push({
            nodeName,
            nodeType: element.type,
            msg: '【' + nodeName + '】' + t('中输入参数中有目标变量未配置，请确认'),
            validate: false,
          });
        }
      });
    }
  }
  // 多实例
  else if (
    element.hasOwnProperty('callActivityType') &&
    element['callActivityType'] == CallActivityType.MULTIPLE
  ) {
    if (element['originatorType'] == MemberType.FORM_FIELD && element['originatorConfig'] == '') {
      validateProcess.push({
        nodeName,
        nodeType: element.type,
        msg: '【' + nodeName + '】' + t('中启用了外部流程，但是未配置发起人，请确认'),
        validate: false,
      });
    }

    //百分比
    if (element['finishType'] == FinishType.PERCENTAGE && !element['percentOf']) {
      validateProcess.push({
        nodeName,
        nodeType: element.type,
        msg: '【' + nodeName + '】' + t(' 外部流程设置了百分比，但是未填写百分比数值，请确认！'),
        validate: false,
      });
    }
  }
  return validateProcess;
}

function basicNode(element: ChildNodeItem): ValidateProcessConfigs {
  const validateProcess: ValidateProcessConfigs = [];
  const nodeName = element.name ? element.name : getNodeName(element.type);
  if (element.name == '' && element.type !== BpmnNodeKey.SEQUENCEFLOW) {
    validateProcess.push({
      nodeName,
      nodeType: element.type,
      msg: t('流程模板中') + nodeName + t('节点名称未填写，请确认'),
      validate: false,
    });
  }
  return validateProcess;
}
// 参数操作
function paramsNode(element: ChildNodeItem): ValidateProcessConfigs {
  const nodeName = element.name ? element.name : getNodeName(element.type);
  const validateProcess: ValidateProcessConfigs = [];
  if (element.hasOwnProperty('assignmentConfig')) {
    const assignmentConfig: AssignmentConfig = element['assignmentConfig'];
    // 表单赋值
    if (assignmentConfig.formAssignmentConfigs.length > 0) {
      assignmentConfig.formAssignmentConfigs.forEach((item) => {
        if (item.source == '') {
          validateProcess.push({
            nodeName,
            nodeType: element.type,
            msg: '【' + nodeName + '】' + t('-参数操作-表单赋值中未选择赋值来源，请确认'),
            validate: false,
          });
        }
        if (!item.target || item.target.formId == '') {
          validateProcess.push({
            nodeName,
            nodeType: element.type,
            msg: '【' + nodeName + '】' + t('-参数操作-表单赋值中未选择目标表单字段值，请确认'),
            validate: false,
          });
        }
      });
    }
    // 参数赋值
    if (assignmentConfig.paramAssignmentConfigs.length > 0) {
      assignmentConfig.paramAssignmentConfigs.forEach((item: ParamAssignmentConfig) => {
        if (item.target == '') {
          validateProcess.push({
            nodeName,
            nodeType: element.type,
            msg: '【' + nodeName + '】' + t('-参数操作-参数赋值中未选择赋值来源，请确认'),
            validate: false,
          });
        }
        if (
          (item.type == ParamType.VALUE && item.value == '') ||
          (item.type == ParamType.VARIABLE && item.varValue == '') ||
          (item.type == ParamType.API && item.apiConfig.id == '') ||
          (item.type == ParamType.FORM_DATA && item.formConfig.formId == '')
        ) {
          validateProcess.push({
            nodeName,
            nodeType: element.type,
            msg: '【' + nodeName + '】' + t('-参数操作-参数赋值中未选择目标参数变量值，请确认'),
            validate: false,
          });
        }
      });
    }
  }

  return validateProcess;
}
