import { defHttp } from '/@/utils/http/axios';

import { ErrorMessageMode } from '/#/axios';
import {
  SalePageListParams,
  SalePageListModel,
  SalePageListResultModel,
  SaleAddParamsModel,
  SaleUpdateParamsModel,
  SaleInfoModel,
  SaleCheckInfoModel,
} from './model';

enum Api {
  Page = '/caseErpSale/caseErpSale/page',
  List = '/caseErpSale/caseErpSale/list',
  Sale = '/caseErpSale/caseErpSale',
  Info = '/caseErpSale/caseErpSale/info',
  Check = '/caseErpSale/caseErpSale/check-info',
  Code = '/caseErpSale/caseErpSale/code',
  Export = '/caseErpSale/caseErpSale/export',
  SaleExport = '/caseErpSale/caseErpSale/export-sale-detail',
  InvoiceExport = '/caseErpSale/caseErpSale/export-sale-invoice',
  LogExport = '/caseErpSale/caseErpSale/export-sale-log',
  OutExport = '/caseErpSale/caseErpSale/export-sale-out',
  ProductExport = '/caseErpSale/caseErpSale/export-sale-product',
  CollectionExport = '/caseErpSale/caseErpSale/export-sale-collection',
}

/**
 * @description: 查询销售订单（分页）
 */
export async function getSalePageList(
  params?: SalePageListParams,
  mode: ErrorMessageMode = 'modal',
) {
  return defHttp.get<SalePageListResultModel>(
    {
      url: Api.Page,
      params,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 查询销售订单（不分页）
 */
export async function getSaleList(mode: ErrorMessageMode = 'modal') {
  return defHttp.get<SalePageListModel>(
    {
      url: Api.List,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 新增销售订单
 */
export async function addSale(params: SaleAddParamsModel, mode: ErrorMessageMode = 'modal') {
  return defHttp.post<number>(
    {
      url: Api.Sale,
      params,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 修改销售订单
 */
export async function updateSale(params: SaleUpdateParamsModel, mode: ErrorMessageMode = 'modal') {
  return defHttp.put<number>(
    {
      url: Api.Sale,
      params,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 删除销售订单
 */
export async function deleteSale(id: string, mode: ErrorMessageMode = 'modal') {
  return defHttp.delete<number>(
    {
      url: Api.Sale,
      data: id,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 查询销售订单详情
 */
export async function getSaleInfo(id: string, mode: ErrorMessageMode = 'modal') {
  return defHttp.get<SaleInfoModel>(
    {
      url: Api.Info,
      params: { id },
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 查看销售订单详情
 */
export async function getSaleCheckInfo(id: string, mode: ErrorMessageMode = 'modal') {
  return defHttp.get<SaleCheckInfoModel>(
    {
      url: Api.Check,
      params: { id },
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 获取销售订单编码
 */
export async function getSaleCode(mode: ErrorMessageMode = 'modal') {
  return defHttp.get<string>(
    {
      url: Api.Code,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 导出
 */
export async function exportInfo(ids: string[], mode: ErrorMessageMode = 'modal') {
  return defHttp.download(
    {
      url: Api.Export,
      method: 'POST',
      params: ids,
      responseType: 'blob',
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 导出订单信息
 */
export async function exportSaleInfo(params, mode: ErrorMessageMode = 'modal') {
  return defHttp.download(
    {
      url: Api.SaleExport,
      method: 'POST',
      params,
      responseType: 'blob',
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 导出出库记录
 */
export async function exportOutstoreInfo(params, mode: ErrorMessageMode = 'modal') {
  return defHttp.download(
    {
      url: Api.OutExport,
      method: 'POST',
      params,
      responseType: 'blob',
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 导出开票记录
 */
export async function exportInvoiceInfo(params, mode: ErrorMessageMode = 'modal') {
  return defHttp.download(
    {
      url: Api.InvoiceExport,
      method: 'POST',
      params,
      responseType: 'blob',
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 导出生产记录
 */
export async function exportProductInfo(params, mode: ErrorMessageMode = 'modal') {
  return defHttp.download(
    {
      url: Api.ProductExport,
      method: 'POST',
      params,
      responseType: 'blob',
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 导出收款记录
 */
export async function exportCollectionInfo(params, mode: ErrorMessageMode = 'modal') {
  return defHttp.download(
    {
      url: Api.CollectionExport,
      method: 'POST',
      params,
      responseType: 'blob',
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 导出操作记录
 */
export async function exportLogInfo(params, mode: ErrorMessageMode = 'modal') {
  return defHttp.download(
    {
      url: Api.LogExport,
      method: 'POST',
      params,
      responseType: 'blob',
    },
    {
      errorMessageMode: mode,
    },
  );
}
