import { defHttp } from '/@/utils/http/axios';

import { ErrorMessageMode } from '/#/axios';
import {
  PropertyPageListSearchModel,
  PropertyAddParamsModel,
  PropertyUpdateParamsModel,
  PropertyPageListResultModel,
} from './model';

enum Api {
  Page = '/caseErpMaterial/caseErpMaterialProperty/page',
  List = '/caseErpMaterial/caseErpMaterialProperty/list',
  Property = '/caseErpMaterial/caseErpMaterialProperty',
  Info = '/caseErpMaterial/caseErpMaterialProperty/info',
  Export = '/caseErpMaterial/caseErpMaterialProperty/export',
  Import = '/caseErpMaterial/caseErpMaterialProperty/import',
}

/**
 * @description: 查询物料属性（分页）
 */
export async function getPropertylPageList(
  params?: PropertyPageListSearchModel,
  mode: ErrorMessageMode = 'modal',
) {
  return defHttp.get<PropertyPageListResultModel>(
    {
      url: Api.Page,
      params,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 查询物料属性（不分页）
 */
export async function getPropertylList(
  params?: PropertyPageListSearchModel,
  mode: ErrorMessageMode = 'modal',
) {
  return defHttp.get<PropertyPageListResultModel>(
    {
      url: Api.List,
      params,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 新增物料属性
 */
export async function addPropertyList(
  type: PropertyAddParamsModel,
  mode: ErrorMessageMode = 'modal',
) {
  return defHttp.post<number>(
    {
      url: Api.Property,
      params: type,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 修改物料属性
 */
export async function updatePropertyList(
  type: PropertyUpdateParamsModel,
  mode: ErrorMessageMode = 'modal',
) {
  return defHttp.put<number>(
    {
      url: Api.Property,
      params: type,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 删除物料属性
 */
export async function deletePropertyList(id: string, mode: ErrorMessageMode = 'modal') {
  return defHttp.delete<number>(
    {
      url: Api.Property,
      data: id,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 查询物料属性
 */
export async function getPropertyListInfo(id: string, mode: ErrorMessageMode = 'modal') {
  return defHttp.get<PropertyAddParamsModel>(
    {
      url: Api.Info,
      params: { id },
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 导出
 */
export async function exportInfo(ids: string[], mode: ErrorMessageMode = 'modal') {
  return defHttp.download(
    {
      url: Api.Export,
      method: 'POST',
      params: ids,
      responseType: 'blob',
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 下载模板
 */
export async function downloadTemplate(mode: ErrorMessageMode = 'modal') {
  return defHttp.download(
    {
      url: Api.Export,
      method: 'GET',
      responseType: 'blob',
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 导入
 */
export async function importInfo(params, mode: ErrorMessageMode = 'modal') {
  return defHttp.post(
    {
      url: Api.Import,
      params,
    },
    {
      errorMessageMode: mode,
    },
  );
}
