import { defHttp } from '/@/utils/http/axios';

import { ErrorMessageMode } from '/#/axios';
import {
  MaterialPageListSearchModel,
  MaterialAddParamsModel,
  MaterialUpdateParamsModel,
  MaterialPageListResultModel,
  MaterialInfoParams,
  PurchaseModel,
  InStoreModel,
  OutStoreModel,
  LogModel,
  MaterialStockParamsModel,
  MaterialHistoryPageListSearchModel,
  MaterialHistoryPageListResultModel,
} from './model';

enum Api {
  Page = '/caseErpMaterial/caseErpMaterial/page',
  List = '/caseErpMaterial/caseErpMaterial/list',
  Material = '/caseErpMaterial/caseErpMaterial',
  Info = '/caseErpMaterial/caseErpMaterial/info',
  Export = '/caseErpMaterial/caseErpMaterial/export',
  Import = '/caseErpMaterial/caseErpMaterial/import',
  Code = '/caseErpMaterial/caseErpMaterial/codeNumber',
  Purchase = '/caseErpMaterial/caseErpMaterial/purchase',
  InStore = '/caseErpMaterial/caseErpMaterial/in-store',
  OutStore = '/caseErpMaterial/caseErpMaterial/out-store',
  Log = '/caseErpMaterial/caseErpMaterial/log',
  Stock = '/caseErpMaterial/caseErpMaterial/stock-count',
  History = '/caseErpMaterial/caseErpMaterial/material-history-page',
}

/**
 * @description: 查询物料清单（分页）
 */
export async function getMaterialPageList(
  params?: MaterialPageListSearchModel,
  mode: ErrorMessageMode = 'modal',
) {
  return defHttp.get<MaterialPageListResultModel>(
    {
      url: Api.Page,
      params,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 查询物料清单（不分页）
 */
export async function getMaterialList(
  params?: MaterialPageListSearchModel,
  mode: ErrorMessageMode = 'modal',
) {
  return defHttp.get<MaterialPageListResultModel>(
    {
      url: Api.List,
      params,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 新增物料信息
 */
export async function addMaterialList(
  type: MaterialAddParamsModel,
  mode: ErrorMessageMode = 'modal',
) {
  return defHttp.post<number>(
    {
      url: Api.Material,
      params: type,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 修改物料信息
 */
export async function updateMaterialList(
  type: MaterialUpdateParamsModel,
  mode: ErrorMessageMode = 'modal',
) {
  return defHttp.put<number>(
    {
      url: Api.Material,
      params: type,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 删除物料信息
 */
export async function deleteMaterialList(id: string, mode: ErrorMessageMode = 'modal') {
  return defHttp.delete<number>(
    {
      url: Api.Material,
      data: id,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 查询物料信息详情
 */
export async function getMaterialListInfo(id: string, mode: ErrorMessageMode = 'modal') {
  return defHttp.get<MaterialAddParamsModel>(
    {
      url: Api.Info,
      params: { id },
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 查询采购记录详情
 */
export async function getPurchaseInfo(
  params: MaterialInfoParams,
  mode: ErrorMessageMode = 'modal',
) {
  return defHttp.get<PurchaseModel[]>(
    {
      url: Api.Purchase,
      params,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 查询入库记录详情
 */
export async function getInStoreInfo(params: MaterialInfoParams, mode: ErrorMessageMode = 'modal') {
  return defHttp.get<InStoreModel[]>(
    {
      url: Api.InStore,
      params,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 查询出库记录详情
 */
export async function getOutStoreInfo(
  params: MaterialInfoParams,
  mode: ErrorMessageMode = 'modal',
) {
  return defHttp.get<OutStoreModel[]>(
    {
      url: Api.OutStore,
      params,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 查询操作记录详情
 */
export async function getLogInfo(params: MaterialInfoParams, mode: ErrorMessageMode = 'modal') {
  return defHttp.get<LogModel[]>(
    {
      url: Api.Log,
      params,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 导出
 */
export async function exportInfo(ids: string[], mode: ErrorMessageMode = 'modal') {
  return defHttp.download(
    {
      url: Api.Export,
      method: 'POST',
      params: ids,
      responseType: 'blob',
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 下载模板
 */
export async function downloadTemplate(mode: ErrorMessageMode = 'modal') {
  return defHttp.download(
    {
      url: Api.Export,
      method: 'GET',
      responseType: 'blob',
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 导入
 */
export async function importInfo(params, mode: ErrorMessageMode = 'modal') {
  return defHttp.post(
    {
      url: Api.Import,
      params,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 获取物料编码
 */
export async function getMaterialCode(mode: ErrorMessageMode = 'modal') {
  return defHttp.get<string>(
    {
      url: Api.Code,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 修改库存盘点、报废
 */
export async function updateMaterialStock(
  params: MaterialStockParamsModel,
  mode: ErrorMessageMode = 'modal',
) {
  return defHttp.put<number>(
    {
      url: Api.Stock,
      params,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 查询历史记录明细（分页）
 */
export async function getMaterialHistoryPageList(
  params?: MaterialHistoryPageListSearchModel,
  mode: ErrorMessageMode = 'modal',
) {
  return defHttp.get<MaterialHistoryPageListResultModel>(
    {
      url: Api.History,
      params,
    },
    {
      errorMessageMode: mode,
    },
  );
}
