import { defHttp } from '/@/utils/http/axios';

import { ErrorMessageMode } from '/#/axios';
import {
  CategoryPageListSearchModel,
  CategoryAddParamsModel,
  CategoryUpdateParamsModel,
  CategoryPageListResultModel,
} from './model';

enum Api {
  Page = '/caseErpMaterial/caseErpMaterialClasses/page',
  List = '/caseErpMaterial/caseErpMaterialClasses/list',
  Category = '/caseErpMaterial/caseErpMaterialClasses',
  Info = '/caseErpMaterial/caseErpMaterialClasses/info',
  Export = '/caseErpMaterial/caseErpMaterialClasses/export',
  Import = '/caseErpMaterial/caseErpMaterialClasses/import',
}

/**
 * @description: 查询物料类别（分页）
 */
export async function getCategoryPageList(
  params?: CategoryPageListSearchModel,
  mode: ErrorMessageMode = 'modal',
) {
  return defHttp.get<CategoryPageListResultModel>(
    {
      url: Api.Page,
      params,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 查询物料类别（不分页）
 */
export async function getCategoryList(
  params?: CategoryPageListSearchModel,
  mode: ErrorMessageMode = 'modal',
) {
  return defHttp.get<CategoryPageListResultModel>(
    {
      url: Api.List,
      params,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 新增物料类别
 */
export async function addCategoryList(
  type: CategoryAddParamsModel,
  mode: ErrorMessageMode = 'modal',
) {
  return defHttp.post<number>(
    {
      url: Api.Category,
      params: type,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 修改物料类别
 */
export async function updateCategoryList(
  type: CategoryUpdateParamsModel,
  mode: ErrorMessageMode = 'modal',
) {
  return defHttp.put<number>(
    {
      url: Api.Category,
      params: type,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 删除物料类别
 */
export async function deleteCategoryList(id: string, mode: ErrorMessageMode = 'modal') {
  return defHttp.delete<number>(
    {
      url: Api.Category,
      data: id,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 查询物料类别详情
 */
export async function getCategoryListInfo(id: string, mode: ErrorMessageMode = 'modal') {
  return defHttp.get<CategoryAddParamsModel>(
    {
      url: Api.Info,
      params: { id },
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 导出
 */
export async function exportInfo(ids: string[], mode: ErrorMessageMode = 'modal') {
  return defHttp.download(
    {
      url: Api.Export,
      method: 'POST',
      params: ids,
      responseType: 'blob',
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 下载模板
 */
export async function downloadTemplate(mode: ErrorMessageMode = 'modal') {
  return defHttp.download(
    {
      url: Api.Export,
      method: 'GET',
      responseType: 'blob',
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 导入
 */
export async function importInfo(params, mode: ErrorMessageMode = 'modal') {
  return defHttp.post(
    {
      url: Api.Import,
      params,
    },
    {
      errorMessageMode: mode,
    },
  );
}
