import { defHttp } from '/@/utils/http/axios';

import { ErrorMessageMode } from '/#/axios';
import {
  TypePageListParams,
  TypePageListSearchModel,
  TypeAddParamsModel,
  TypeUpdateParamsModel,
  TypePageListResultModel,
  TypeListModel,
} from './model';

enum Api {
  Page = '/caseErpUnit/caseErpUnitType/page',
  List = '/caseErpUnit/caseErpUnitType/list',
  Type = '/caseErpUnit/caseErpUnitType',
  Info = '/caseErpUnit/caseErpUnitType/info',
  Export = '/caseErpUnit/caseErpUnitType/export',
  Import = '/caseErpUnit/caseErpUnitType/import',
}

/**
 * @description: 查询单位类型列表（分页）
 */
export async function getUnitTypePageList(
  params?: TypePageListParams,
  mode: ErrorMessageMode = 'modal',
) {
  return defHttp.get<TypePageListResultModel>(
    {
      url: Api.Page,
      params,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 查询单位类型列表（不分页）
 */
export async function getUnitTypeList(
  params?: TypePageListSearchModel,
  mode: ErrorMessageMode = 'modal',
) {
  return defHttp.get<TypeListModel[]>(
    {
      url: Api.List,
      params,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 新增单位类型
 */
export async function addUnitType(type: TypeAddParamsModel, mode: ErrorMessageMode = 'modal') {
  return defHttp.post<number>(
    {
      url: Api.Type,
      params: type,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 修改单位类型
 */
export async function updateUnitType(
  type: TypeUpdateParamsModel,
  mode: ErrorMessageMode = 'modal',
) {
  return defHttp.put<number>(
    {
      url: Api.Type,
      params: type,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 删除单位类型（批量删除）
 */
export async function deleteUnitType(ids: string[], mode: ErrorMessageMode = 'modal') {
  return defHttp.delete<number>(
    {
      url: Api.Type,
      data: ids,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 查询单位类型详情
 */
export async function getUnitTypeInfo(id: string, mode: ErrorMessageMode = 'modal') {
  return defHttp.get<TypeAddParamsModel>(
    {
      url: Api.Info,
      params: { id },
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 导出
 */
export async function exportInfo(ids: string[], mode: ErrorMessageMode = 'modal') {
  return defHttp.download(
    {
      url: Api.Export,
      method: 'POST',
      params: ids,
      responseType: 'blob',
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 下载模板
 */
export async function downloadTemplate(mode: ErrorMessageMode = 'modal') {
  return defHttp.download(
    {
      url: Api.Export,
      method: 'GET',
      responseType: 'blob',
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 导入
 */
export async function importInfo(params, mode: ErrorMessageMode = 'modal') {
  return defHttp.post(
    {
      url: Api.Import,
      params,
    },
    {
      errorMessageMode: mode,
    },
  );
}
