import { defHttp } from '/@/utils/http/axios';

import { ErrorMessageMode } from '/#/axios';
import {
  PurchaseOrderPageListSearchModel,
  PurchaseOrderPageListResultModel,
  PurchaseOrderListModel,
  PurchaseOrderAddParamsModel,
  PurchaseOrderUpdateParamsModel,
  PurchaseOrderCheckInfoModel,
} from './model';

enum Api {
  Page = '/caseErpPurchase/caseErpPurchase/page',
  List = '/caseErpPurchase/caseErpPurchase/list',
  Purchase = '/caseErpPurchase/caseErpPurchase',
  Info = '/caseErpPurchase/caseErpPurchase/info',
  Check = '/caseErpPurchase/caseErpPurchase/check-info',
  Code = '/caseErpSale/caseErpSale/code',
  Record = '/workflow/execute/process/all-record',
  Export = '/caseErpPurchase/caseErpPurchase/export',
  PurchaseExport = '/caseErpPurchase/caseErpPurchase/export-purchase-detail',
  TicketExport = '/caseErpPurchase/caseErpPurchase/export-purchase-ticket',
  LogExport = '/caseErpPurchase/caseErpPurchase/export-purchase-log',
  InExport = '/caseErpPurchase/caseErpPurchase/export-purchase-in',
  PayExport = '/caseErpPurchase/caseErpPurchase/export-purchase-pay',
}

/**
 * @description: 查询采购订单（分页）
 */
export async function getPurchasePageList(
  params?: PurchaseOrderPageListSearchModel,
  mode: ErrorMessageMode = 'modal',
) {
  return defHttp.get<PurchaseOrderPageListResultModel>(
    {
      url: Api.Page,
      params,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 查询采购订单（不分页）
 */
export async function getPurchaseList(mode: ErrorMessageMode = 'modal') {
  return defHttp.get<PurchaseOrderListModel[]>(
    {
      url: Api.List,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 新增采购订单
 */
export async function addPurchase(
  params: PurchaseOrderAddParamsModel,
  mode: ErrorMessageMode = 'modal',
) {
  return defHttp.post<number>(
    {
      url: Api.Purchase,
      params,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 修改采购订单
 */
export async function updatePurchase(
  params: PurchaseOrderUpdateParamsModel,
  mode: ErrorMessageMode = 'modal',
) {
  return defHttp.put<number>(
    {
      url: Api.Purchase,
      params,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 删除采购订单
 */
export async function deletePurchase(id: string, mode: ErrorMessageMode = 'modal') {
  return defHttp.delete<number>(
    {
      url: Api.Purchase,
      data: id,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 查询采购订单详情
 */
export async function getPurchaseInfo(id: string, mode: ErrorMessageMode = 'modal') {
  return defHttp.get<PurchaseOrderrUpdateParamsModel>(
    {
      url: Api.Info,
      params: { id },
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 查看采购订单详情
 */
export async function getPurchaseCheckInfo(id: string, mode: ErrorMessageMode = 'modal') {
  return defHttp.get<PurchaseOrderCheckInfoModel>(
    {
      url: Api.Check,
      params: { id },
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 获取销售订单编码
 */
export async function getSaleCode(mode: ErrorMessageMode = 'modal') {
  return defHttp.get<string>(
    {
      url: Api.Code,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 导出
 */
export async function exportInfo(ids: string[], mode: ErrorMessageMode = 'modal') {
  return defHttp.download(
    {
      url: Api.Export,
      method: 'POST',
      params: ids,
      responseType: 'blob',
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 导出采购信息
 */
export async function exportPurchaseInfo(params, mode: ErrorMessageMode = 'modal') {
  return defHttp.download(
    {
      url: Api.PurchaseExport,
      method: 'POST',
      params,
      responseType: 'blob',
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 导出入库记录
 */
export async function exportInstoreInfo(params, mode: ErrorMessageMode = 'modal') {
  return defHttp.download(
    {
      url: Api.InExport,
      method: 'POST',
      params,
      responseType: 'blob',
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 导出到票记录
 */
export async function exportTicketInfo(params, mode: ErrorMessageMode = 'modal') {
  return defHttp.download(
    {
      url: Api.TicketExport,
      method: 'POST',
      params,
      responseType: 'blob',
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 导出付款记录
 */
export async function exportPayInfo(params, mode: ErrorMessageMode = 'modal') {
  return defHttp.download(
    {
      url: Api.PayExport,
      method: 'POST',
      params,
      responseType: 'blob',
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 导出操作记录
 */
export async function exportLogInfo(params, mode: ErrorMessageMode = 'modal') {
  return defHttp.download(
    {
      url: Api.LogExport,
      method: 'POST',
      params,
      responseType: 'blob',
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 查询审批记录
 */
export async function getRecordList(processId: string, mode: ErrorMessageMode = 'modal') {
  return defHttp.get(
    {
      url: Api.Record,
      params: { processId },
    },
    {
      errorMessageMode: mode,
    },
  );
}
