import { defHttp } from '/@/utils/http/axios';

import { ErrorMessageMode } from '/#/axios';
import {
  CustomerPageListSearchModel,
  CustomerPageListResultModel,
  CustomerListModel,
  CustomerAddParamsModel,
  CustomerUpdateParamsModel,
  CustomerDetailInfoModel,
  FollowAddParamsModel,
  FollowUpdateParamsModel,
  ContactsAddParamsModel,
  ContactsUpdateParamsModel,
  TransferParamsModel,
  CustomerInfoModel,
} from './model';

enum Api {
  List = '/caseErpCustomer/caseErpCustomer/list',
  Page = '/caseErpCustomer/caseErpCustomer/page',
  Info = '/caseErpCustomer/caseErpCustomer/info',
  Customer = '/caseErpCustomer/caseErpCustomer',
  Follow = '/caseErpCustomer/caseErpCustomerFollow',
  Contacts = '/caseErpCustomer/caseErpCustomerContacts',
  ContactsInfo = '/caseErpCustomer/caseErpCustomerContacts/info',
  DetailInfo = '/caseErpCustomer/caseErpCustomer/detail-info',
  FollowInfo = '/caseErpCustomer/caseErpCustomerFollow/info',
  TransferUser = '/caseErpCustomer/caseErpCustomer/transfer',
  Common = '/caseErpCustomer/caseErpCustomer/common',
  Export = '/caseErpCustomer/caseErpCustomer/export',
}

/**
 * @description: 查询客户列表（不分页）
 */
export async function getCustomerList(mode: ErrorMessageMode = 'modal') {
  return defHttp.get<CustomerListModel>(
    {
      url: Api.List,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 查询客户列表（分页）
 */
export async function getCustomerPageList(
  params: CustomerPageListSearchModel,
  mode: ErrorMessageMode = 'modal',
) {
  return defHttp.get<CustomerPageListResultModel>(
    {
      url: Api.Page,
      params,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 新增客户
 */
export async function addCustomer(
  params: CustomerAddParamsModel,
  mode: ErrorMessageMode = 'modal',
) {
  return defHttp.post<number>(
    {
      url: Api.Customer,
      params,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 修改客户
 */
export async function updateCustomer(
  params: CustomerUpdateParamsModel,
  mode: ErrorMessageMode = 'modal',
) {
  return defHttp.put<number>(
    {
      url: Api.Customer,
      params,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 删除客户
 */
export async function deleteCustomer(id: string, mode: ErrorMessageMode = 'modal') {
  return defHttp.delete<number>(
    {
      url: Api.Customer,
      data: id,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 查询客户详情
 */
export async function getCustomerInfo(id: string, mode: ErrorMessageMode = 'modal') {
  return defHttp.get<CustomerInfoModel>(
    {
      url: Api.Info,
      params: { id },
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 查询客户列表详情
 */
export async function getDetailInfo(id: string, mode: ErrorMessageMode = 'modal') {
  return defHttp.get<CustomerDetailInfoModel>(
    {
      url: Api.DetailInfo,
      params: { id },
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 新增跟进客户
 */
export async function addFollowCustomer(
  params: FollowAddParamsModel,
  mode: ErrorMessageMode = 'modal',
) {
  return defHttp.post<number>(
    {
      url: Api.Follow,
      params,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 编辑跟进客户
 */
export async function updateFollowCustomer(
  params: FollowUpdateParamsModel,
  mode: ErrorMessageMode = 'modal',
) {
  return defHttp.put<number>(
    {
      url: Api.Follow,
      params,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 删除跟进客户
 */
export async function deleteFollowCustomer(id: string, mode: ErrorMessageMode = 'modal') {
  return defHttp.delete<number>(
    {
      url: Api.Follow,
      data: id,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 查询跟进客户列表详情
 */
export async function getFollowInfo(id: string, mode: ErrorMessageMode = 'modal') {
  return defHttp.get<FollowAddParamsModel>(
    {
      url: Api.FollowInfo,
      params: { id },
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 新增联系人
 */
export async function addContacts(
  params: ContactsAddParamsModel,
  mode: ErrorMessageMode = 'modal',
) {
  return defHttp.post<number>(
    {
      url: Api.Contacts,
      params,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 编辑联系人
 */
export async function updateContacts(
  params: ContactsUpdateParamsModel,
  mode: ErrorMessageMode = 'modal',
) {
  return defHttp.put<number>(
    {
      url: Api.Contacts,
      params,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 删除联系人
 */
export async function deleteContacts(id: string, mode: ErrorMessageMode = 'modal') {
  return defHttp.delete<number>(
    {
      url: Api.Contacts,
      data: id,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 查询联系人详情
 */
export async function getContactsInfo(id: string, mode: ErrorMessageMode = 'modal') {
  return defHttp.get<ContactsUpdateParamsModel>(
    {
      url: Api.ContactsInfo,
      params: { id },
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 转移客户
 */
export async function transferUser(params: TransferParamsModel, mode: ErrorMessageMode = 'modal') {
  return defHttp.put<number>(
    {
      url: Api.TransferUser,
      params,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 移入公海
 */
export async function transferCommon(ids: string[], mode: ErrorMessageMode = 'modal') {
  return defHttp.put<number>(
    {
      url: Api.Common,
      params: ids,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 导出
 */
export async function exportInfo(ids?: string[], mode: ErrorMessageMode = 'modal') {
  return defHttp.download(
    {
      url: Api.Export,
      method: 'POST',
      params: ids,
      responseType: 'blob',
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 下载模板
 */
export async function downloadTemplate(mode: ErrorMessageMode = 'modal') {
  return defHttp.download(
    {
      url: Api.Export,
      method: 'GET',
      responseType: 'blob',
    },
    {
      errorMessageMode: mode,
    },
  );
}
