import { getFunction } from '@/utils/utils'

/**
 * 处理组件数据传输的核心函数
 * @param {Array} list - 操作列表
 * @param {Object} refList - 引用列表
 * @param {Object} params - 参数对象
 * @param {Object} safe - 安全对象
 */
export default function (list, refList, params, safe) {
	/**
	 * 递归遍历树结构数组
	 * @param {Array} list - 树形数组
	 * @param {Function} fn - 对叶子节点执行的回调函数
	 */
	const arrayTree = (list, fn) => {
		list.forEach(ele => {
			if (ele.children) arrayTree(ele.children)
			else fn(ele)
		})
	}

	// 操作类型处理器映射表
	const typeHandlers = {
		/**
		 * 处理参数或数据类型的操作
		 * @param {Object} ele - 当前元素
		 */
		params: (ele) => {
			let p = {};
			ele.child.forEach(item => {
				p[item.name] = params[item.value]
			})
			ele.index.forEach(index => {
				let { item } = this.contain.findnav(index)
				if (!item) return
				if (item.children) {
					arrayTree(item.children, (child) => {
						refList[child.index].updateData(p);
					});
				} else {
					refList[index].updateData(p);
				}
			})
		},

		/**
		 * 处理数据类型的操作（与params相同的逻辑）
		 * @param {Object} ele - 当前元素
		 */
		data: (ele) => {
			typeHandlers.params(ele);
		},

		/**
		 * 处理组类型的操作
		 * @param {Object} ele - 当前元素
		 */
		group: (ele) => {
			window.$glob.group = ele.group
		},

		/**
		 * 处理链接跳转类型的操作
		 * @param {Object} ele - 当前元素
		 */
		href: (ele) => {
			// 构建参数对象
			let p = {};
			if (ele.child) {
				ele.child.forEach(item => {
					p[item.name] = params[item.value]
				})
			}
			// 使用更简单的URL拼接方式
			const url = new URL(ele.src, window.location.origin);
			Object.keys(p).forEach(key => {
				url.searchParams.set(key, p[key]);
			});
			const urlWithParams = url.toString();
			
			ele.target ? window.open(urlWithParams) : (location.href = urlWithParams);
		},

		/**
		 * 处理对话框类型的操作
		 * @param {Object} ele - 当前元素
		 */
		dialog: (ele) => {
			const oldGroup = window.$glob.group;
			let dialogList = document.getElementsByClassName('dialog')
			let dialog = document.createElement('div');
			const zIndex = 10000 + dialogList.length;
			dialog.className = 'dialog'
			dialog.style.zIndex = zIndex
			dialog.innerHTML = `<div class="dialog__title">${ele.title}</div>`
			document.getElementById('container').append(dialog)
			
			// 对话框点击事件处理
			dialog.addEventListener('click', () => {
				const showCallback = (index) => {
					let refObj = this.container.getItemRef(index)
					if (refObj) refObj.object.display = true
				}
				ele.index.forEach(index => {
					let { item } = this.contain.findnav(index)
					if (item.children) {
						arrayTree(item.children, (child) => {
							showCallback(child.index)
						})
					} else {
						showCallback(index)
					}
				})
				window.$glob.group = oldGroup
				dialog.remove()
			})

			// 隐藏组件的回调函数
			const hideCallback = (list) => {
				list.forEach(index => {
					let { item } = this.contain.findnav(index)
					if (!item) return
					const hideItemCallback = (index) => {
						let styleObj = this.container.getListRef(index)
						let refObj = this.container.getItemRef(index)
						if (styleObj && refObj) {
							refObj.object.display = false;
							setTimeout(() => {
								styleObj = styleObj.$el.style;
								styleObj.zIndex = zIndex + 1;
							}, 0)
						}
					}
					if (item.children) {
						arrayTree(item.children, (child) => {
							hideItemCallback(child.index)
						})
					} else {
						hideItemCallback(index)
					}
				})
			}
			
			hideCallback(ele.index)
			if (ele.group) {
				window.$glob.group = ele.group
				setTimeout(() => {
					let list = this.contain.allList.filter(item => item.group == ele.group).map(item => item.index);
					hideCallback(list)
				})
			}
		},

		/**
		 * 处理显示控制类型的操作
		 * @param {Object} ele - 当前元素
		 */
		display: (ele) => {
			ele.index.forEach(index => {
				let { item } = this.contain.findnav(index)
				if (!item) return
				const displayCallback = (index) => {
					let refObj = this.container.getItemRef(index)
					if (refObj) {
						if (ele.displayType == '') {
							refObj.object.display = !refObj.object.display
						} else {
							refObj.object.display = ele.displayType
						}
					}
				}
				if (item.children) {
					arrayTree(item.children, (child) => {
						displayCallback(child.index)
					})
				} else {
					displayCallback(index)
				}
			})
		},

		/**
		 * 处理自定义函数类型的操作
		 * @param {Object} ele - 当前元素
		 */
		fun: (ele) => {
			let fun = getFunction(ele.fun, true);
			fun(params, refList, safe)
		}
	};

	// 遍历操作列表并执行相应的处理器
	list.forEach(ele => {
		const handler = typeHandlers[ele.type];
		if (handler) {
			handler(ele);
		}
	})
}