import { defHttp } from '/@/utils/http/axios';

import { ErrorMessageMode } from '/#/axios';
import {
  DevicePageListSearchModel,
  DeviceAddParamsModel,
  DeviceUpdateParamsModel,
  DevicePageListResultModel,
  DeviceListModel,
} from './model';

enum Api {
  Page = '/caseErpDevice/caseErpDeviceInfo/page',
  Device = '/caseErpDevice/caseErpDeviceInfo',
  Info = '/caseErpDevice/caseErpDeviceInfo/info',
  Export = '/caseErpDevice/caseErpDeviceInfo/export',
  Check = '/caseErpDevice/caseErpDeviceInfo/check',
  ByTypeId = '/caseErpDevice/caseErpDeviceInfo/get-by-type-id',
}

/**
 * @description: 查询设备信息（分页）
 */
export async function getDeviceInfoPageList(
  params?: DevicePageListSearchModel,
  mode: ErrorMessageMode = 'modal',
) {
  return defHttp.get<DevicePageListResultModel>(
    {
      url: Api.Page,
      params,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 新增设备信息
 */
export async function addDeviceInfo(type: DeviceAddParamsModel, mode: ErrorMessageMode = 'modal') {
  return defHttp.post<number>(
    {
      url: Api.Device,
      params: type,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 修改设备信息
 */
export async function updateDeviceInfo(
  params: DeviceUpdateParamsModel,
  mode: ErrorMessageMode = 'modal',
) {
  return defHttp.put<number>(
    {
      url: Api.Device,
      params,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 删除设备信息
 */
export async function deleteDeviceInfo(id: string, mode: ErrorMessageMode = 'modal') {
  return defHttp.delete<number>(
    {
      url: Api.Device,
      data: id,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 查询设备信息详情
 */
export async function getDeviceInfo(id: string, mode: ErrorMessageMode = 'modal') {
  return defHttp.get<DeviceAddParamsModel>(
    {
      url: Api.Info,
      params: { id },
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 导出
 */
export async function exportInfo(ids: string[], mode: ErrorMessageMode = 'modal') {
  return defHttp.download(
    {
      url: Api.Export,
      method: 'POST',
      params: ids,
      responseType: 'blob',
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 查看设备信息
 */
export async function checkDeviceInfo(id: string, mode: ErrorMessageMode = 'modal') {
  return defHttp.get<DeviceListModel>(
    {
      url: Api.Check,
      params: { id },
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 根据设备类型查询设备信息
 */
export async function getDeviceInfoByTypeId(typeId: string, mode: ErrorMessageMode = 'modal') {
  return defHttp.get<DeviceListModel[]>(
    {
      url: Api.ByTypeId,
      params: { typeId },
    },
    {
      errorMessageMode: mode,
    },
  );
}
