import { defHttp } from '/@/utils/http/axios';

import { ErrorMessageMode } from '/#/axios';
import {
  MaterialListSearchModel,
  MaterialAddParamsModel,
  MaterialUpdateParamsModel,
  MaterialListResultModel,
} from './model';

enum Api {
  Page = '/caseErpSupplier/caseErpSupplyMaterial/page',
  Material = '/caseErpSupplier/caseErpSupplyMaterial',
  Info = '/caseErpSupplier/caseErpSupplyMaterial/info',
  Export = '/caseErpSupplier/caseErpSupplyMaterial/export',
  Import = '/caseErpSupplier/caseErpSupplyMaterial/import',
}

/**
 * @description: 查询供货清单（分页）
 */
export async function getSupplyMaterialPageList(
  params?: MaterialListSearchModel,
  mode: ErrorMessageMode = 'modal',
) {
  return defHttp.get<MaterialListResultModel>(
    {
      url: Api.Page,
      params,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 新增供货清单
 */
export async function addSupplyMaterial(
  type: MaterialAddParamsModel,
  mode: ErrorMessageMode = 'modal',
) {
  return defHttp.post<number>(
    {
      url: Api.Material,
      params: type,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 修改供货清单
 */
export async function updateSupplyMaterial(
  type: MaterialUpdateParamsModel,
  mode: ErrorMessageMode = 'modal',
) {
  return defHttp.put<number>(
    {
      url: Api.Material,
      params: type,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 删除供货清单
 */
export async function deleteSupplyMaterial(id: string, mode: ErrorMessageMode = 'modal') {
  return defHttp.delete<number>(
    {
      url: Api.Material,
      data: id,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 查询单位详情
 */
export async function getSupplyMaterialInfo(id: string, mode: ErrorMessageMode = 'modal') {
  return defHttp.get<MaterialAddParamsModel>(
    {
      url: Api.Info,
      params: { id },
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 导出
 */
export async function exportInfo(id: string, mode: ErrorMessageMode = 'modal') {
  return defHttp.download(
    {
      url: Api.Export,
      method: 'POST',
      params: { id },
      responseType: 'blob',
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 下载模板
 */
export async function downloadTemplate(mode: ErrorMessageMode = 'modal') {
  return defHttp.download(
    {
      url: Api.Export,
      method: 'GET',
      responseType: 'blob',
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 导入
 */
export async function importInfo(params, mode: ErrorMessageMode = 'modal') {
  return defHttp.post(
    {
      url: Api.Import,
      params,
    },
    {
      errorMessageMode: mode,
    },
  );
}
