import { defHttp } from '/@/utils/http/axios';

import { ErrorMessageMode } from '/#/axios';
import {
  SupplierPageListSearchModel,
  SupplierPageListResultModel,
  SupplierAddParamsModel,
  SupplierUpdateParamsModel,
  SupplierInfoModel,
  ApplicationAddParamsModel,
} from './model';

enum Api {
  Info = '/caseErpSupplier/caseErpSupplier/info',
  List = '/caseErpSupplier/caseErpSupplier/list',
  Page = '/caseErpSupplier/caseErpSupplier/page',
  Supplier = '/caseErpSupplier/caseErpSupplier',
  RiskInfo = '/caseErpSupplier/caseErpSupplier/risk-info',
  AnnualInfo = '/caseErpSupplier/caseErpSupplier/year-info',
  DetailAnnualInfo = '/caseErpSupplier/caseErpSupplier/detail-year-info',
  TimeInfo = '/caseErpSupplier/caseErpSupplier/confirmation-info',
  AddRisk = '/caseErpSupplier/caseErpSupplier/add-risk',
  Code = '/caseErpSupplier/caseErpSupplier/codeNumber',
  Confirmation = '/caseErpSupplier/caseErpSupplier/confirmation',
  Regain = '/caseErpSupplier/caseErpSupplier/supplier-recover',
  Eliminate = '/caseErpSupplier/caseErpSupplier/supplier-out',
}

/**
 * @description: 查询供应商（不分页）
 */
export async function getSupplierList(mode: ErrorMessageMode = 'modal') {
  return defHttp.get<SupplierPageListResultModel>(
    {
      url: Api.List,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 查询供应商（分页）
 */
export async function getSupplierPageList(
  params: SupplierPageListSearchModel,
  mode: ErrorMessageMode = 'modal',
) {
  return defHttp.get<SupplierPageListResultModel>(
    {
      url: Api.Page,
      params,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 新增供应商
 */
export async function addSupplier(
  params: SupplierAddParamsModel,
  mode: ErrorMessageMode = 'modal',
) {
  return defHttp.post<number>(
    {
      url: Api.Supplier,
      params,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 修改供应商
 */
export async function updateSupplier(
  params: SupplierUpdateParamsModel,
  mode: ErrorMessageMode = 'modal',
) {
  return defHttp.put<number>(
    {
      url: Api.Supplier,
      params,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 删除供应商
 */
export async function deleteSupplier(id: string, mode: ErrorMessageMode = 'modal') {
  return defHttp.delete<number>(
    {
      url: Api.Supplier,
      data: id,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 查询供应商详情
 */
export async function getSupplierInfo(id: string, mode: ErrorMessageMode = 'modal') {
  return defHttp.get<SupplierInfoModel>(
    {
      url: Api.Info,
      params: { id },
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 查询风险评估详情
 */
export async function getRiskInfo(id: string, mode: ErrorMessageMode = 'modal') {
  return defHttp.get(
    {
      url: Api.RiskInfo,
      params: { id },
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 查询年审详情
 */
export async function getAnnualInfo(id: string, mode: ErrorMessageMode = 'modal') {
  return defHttp.get(
    {
      url: Api.AnnualInfo,
      params: { id },
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 根据formalId查询年审详情
 */
export async function getDetailAnnualInfo(id: string, mode: ErrorMessageMode = 'modal') {
  return defHttp.get(
    {
      url: Api.DetailAnnualInfo,
      params: { id },
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 新增风险评估
 */
export async function addRisk(params: SupplierAddParamsModel, mode: ErrorMessageMode = 'modal') {
  return defHttp.post<number>(
    {
      url: Api.AddRisk,
      params,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 获取供应商编码
 */
export async function getSupplierCode(mode: ErrorMessageMode = 'modal') {
  return defHttp.get<string>(
    {
      url: Api.Code,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 转正申请
 */
export async function addApplication(
  params: ApplicationAddParamsModel,
  mode: ErrorMessageMode = 'modal',
) {
  return defHttp.post<number>(
    {
      url: Api.Confirmation,
      params,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 查询供应商详情（转正说明及时间轴）
 */
export async function getSupplierTimeInfo(id: string, mode: ErrorMessageMode = 'modal') {
  return defHttp.get<SupplierInfoModel>(
    {
      url: Api.TimeInfo,
      params: { id },
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 恢复供应商
 */
export async function regainSupplier(
  params: SupplierAddParamsModel,
  mode: ErrorMessageMode = 'modal',
) {
  return defHttp.put<number>(
    {
      url: Api.Regain,
      params,
    },
    {
      errorMessageMode: mode,
    },
  );
}

/**
 * @description: 淘汰供应商
 */
export async function eliminateSupplier(
  params: SupplierAddParamsModel,
  mode: ErrorMessageMode = 'modal',
) {
  return defHttp.put<number>(
    {
      url: Api.Eliminate,
      params,
    },
    {
      errorMessageMode: mode,
    },
  );
}
