import datetime
import math

from odoo import models, fields, api, _
from odoo.exceptions import ValidationError
from odoo.tools.float_utils import float_round
import logging
_logger = logging.getLogger(__name__)


def _get_pd(env, index="Production"):
    return env["decimal.precision"].precision_get(index)


def float_to_time(hours):
    """ 将小时数转换为时间对象. """
    if hours == 24.0:
        return datetime.time.max.replace(microsecond=0)
    fractional, integral = math.modf(hours)
    return datetime.time(int(integral), int(float_round(60 * fractional, precision_digits=0)), 0)


class InheritRokeCreateWorkRecordWizard(models.TransientModel):
    _inherit = "roke.create.work.record.wizard"

    repair_process_id = fields.Many2one("roke.process", string="工序")
    repair_qty = fields.Float(string="返修数", digits='Production', default=0)

    def create_other_order(self, new_record):
        """
        创建报废单、返修单
        生产返修工单
        :return:
        """
        res = super(InheritRokeCreateWorkRecordWizard, self).create_other_order(new_record)
        repair = res.get("repair", False)
        refix_params = self.env['ir.config_parameter'].sudo().get_param("refix_auto_work_order")
        if repair and self.repair_qty and (refix_params == "True" or refix_params is True) and self.repair_process_id:
            repair.is_refix_auto_work_order = refix_params == "True" or refix_params is True
            for v in repair.line_ids:
                repair_work_order_id = self.env["roke.work.order"].create({
                    "process_id": self.repair_process_id.id,
                    "product_id": new_record.work_order_id.product_id.id,
                    "plan_qty": v.qty,
                    "sequence": 1,
                    "planned_start_time": new_record.work_order_id.planned_start_time,
                    "plan_date": new_record.work_order_id.plan_date,
                    "repair_task_id": new_record.pt_id.id or repair.wo_id.repair_task_id.id,
                    "repair_line_id": v.id,
                    "type": "返修"
                })
                data = {
                    "repair_work_order_id": repair_work_order_id,
                    "is_refix_auto_work_order": refix_params == "True" or refix_params is True
                }
                if repair_work_order_id.wo_start_state == '已开工':
                    data.update({"state": "返修中"})
                    repair.write({"state": "返修中"})
                v.write(data)
        return res

    @api.onchange("unqualified_qty")
    def _onchange_unqualified_qty(self):
        """不合格数默认为报废数"""
        return {"value": {"scrap_qty": self.unqualified_qty}}

    @api.onchange("scrap_qty")
    def _onchange_scrap_qty(self):
        """
        1.报废数大于不合格数时：设置报废数等于不合格数；返修数等于0
        2.报废数加返修数大于不合格数时：设置返修数等于不合格数减报废数
        :return:
        """
        if self.scrap_qty > self.unqualified_qty:
            return {"value": {"scrap_qty": self.unqualified_qty}}

    @api.onchange("repair_qty")
    def _onchange_repair_qty(self):
        """
        1.返修数大于不合格数时：设置返修数等于不合格数；报废数等于0
        2.返修数加报废数大于不合格数时：设置报废数等于不合格数减返修数
        :return:
        """
        ...  # 去掉返修和报废的逻辑关系

    def check_create_value(self):
        """
        检查完成数、不合格数、工时数
        :return:
        """
        if not self.finish_qty and not self.unqualified_qty and not self.work_hours and not self.repair_qty:
            return False
        else:
            return True

    def confirm(self, center=False):
        """
        生成报工记录
        :return:
        """
        if self.work_order_id.wo_start_state == "未开工":
            raise ValidationError("当前工单未开工，禁止报工")
        if self.work_order_id.wo_child_type == "main" and self.work_order_id.child_wo_ids:
            raise ValidationError("主工序工单禁止报工")
        if self.work_order_id.state in ["暂停", "强制完工"]:
            raise ValidationError("工单（{}）状态是{}不允许报工".format(self.work_order_id.code, self.work_order_id.state))
        if not self.check_create_value():
            raise ValidationError("完成数，不合格数，工时不能同时为空或0")
        if self.multi and not self.allot_ids:
            raise ValidationError("请填写报工人员。")

        # 校验在此之前的工序报工数
        if self.work_order_id.wo_child_type == "child" and self.work_order_id.main_wo_id:
            previous = self.work_order_id._get_child_previous()
        else:
            previous = self.work_order_id._get_previous()

        ConfigParameter = self.sudo().env['ir.config_parameter']
        report_qty = self.finish_qty  # 待校验的报工数
        wait_qty = previous.finish_qty * self.work_order_id._get_production_multiple(previous)\
            if previous and self.env.user.company_id.freedom_work != "允许" else self.wait_qty
        if self.env.user.company_id.complete_basis != "合格数":
            report_qty = round(report_qty + self.unqualified_qty, _get_pd(self.env))  # 待校验的报工数
        else:
            if self.env.user.company_id.freedom_work != "允许" and previous and previous.finish_qty <= 0:
                raise ValidationError(f"前工序合格数为0，禁止报工。（如想取消该限制，请联系系统管理员将“自由报工”选项设置为“允许”）")
        exceed_plan = self.env.user.company_id.exceed_plan
        exceed_plan_qty = float(ConfigParameter.get_param('exceed.plan.qty', default=0)) if exceed_plan == "允许" else 0
        if exceed_plan != "允许" or (exceed_plan == "允许" and exceed_plan_qty):
            _logger.error(f"报工数量：{str(report_qty)} 计划 : {self.plan_qty} 剩余 : {self.wait_qty} 剩余二 ：{str(wait_qty)}, {exceed_plan_qty}")
            if report_qty > wait_qty + exceed_plan_qty:
                raise ValidationError(f"报工数量：{str(report_qty)} 不可大于可报数量：{str(wait_qty)}。"
                                      f"（如想取消该限制，请联系系统管理员将“超计划报工”选项设置为“允许”）")

        # 校验报工时间是否超出班组报工时间限制
        teams = self.team_id or self.allot_ids.employee_id.team_id or self.employee_id.team_id
        restrict_date_team = teams.filtered(lambda team: team.restrict_work_date)
        if restrict_date_team and not self.env.user.has_group("base.group_system"):
            today = fields.Date.context_today(self)
            last_days = min(restrict_date_team.mapped("last_days"))
            after_days = max(restrict_date_team.mapped("after_days"))
            last_time = datetime.datetime.combine(today - datetime.timedelta(days=last_days), float_to_time(max(restrict_date_team.mapped("last_time"))))
            after_time = datetime.datetime.combine(today + datetime.timedelta(days=after_days), float_to_time(max(restrict_date_team.mapped("after_time"))))
            work_time = self.work_time + datetime.timedelta(hours=8)
            if work_time < last_time or work_time > after_time:
                raise ValidationError("报工时间错误，当前人员可报工时间范围：%s 到 %s。" % (str(last_time), str(after_time)))
        # 校验在此之前的工序报工数
        if previous and self.env.user.company_id.freedom_work != "允许":
            self._check_freedom_work()
        # 创建报工记录
        employee_ids, allot_list = self._get_employee_allot()
        new_record = self.env["roke.work.record"]
        try:
            new_record = self.env["roke.work.record"].create(self._get_work_record_dict(employee_ids, allot_list))
            self.create_other_order(new_record)
            self.write({"wr_id": new_record.id})
            # 工单完工
            self.work_order_id.finish_report_work_order(self.finish_qty, self.unqualified_qty, self.work_hours, finish_time=self.work_time, wr_id=new_record.id)
            # 工装生命周期扣除by:duqinglin
            if new_record.work_center_id and new_record.process_id.is_press:
                qty = new_record.unqualified_qty + new_record.finish_qty
                for tool in new_record.work_center_id.tool_ids:
                    if tool.type_id.name != '模具':
                        continue
                    user_freq = math.ceil(qty / tool.once_qty)
                    tool.write({'use_record': [(0, 0, {'use_life_cycle': user_freq,
                                                       'use_users': self.employee_id.user_id.id,
                                                       'equipment_id': tool.equipment_id.id,
                                                       'use_type': '报工'})]})
                    tool.used_life_cycle = tool.used_life_cycle + user_freq
                    tool.residue_life_cycle = tool.standard_life_cycle - tool.used_life_cycle
        except Exception as e:
            _logger.error("！！！！！报工向导类报错！！！！！")
            _logger.error(e)
            if new_record:
                _logger.error(new_record)
                try:
                    new_record.withdraw()
                except Exception as withdraw_error:
                    new_record.unlink()
            raise e
        return {'type': 'ir.actions.act_window_close'}
