from odoo import models, fields, api, _
from odoo.exceptions import ValidationError, UserError


class InheritWorkOrder(models.Model):
    _inherit = "roke.work.order"

    document_ids = fields.Many2many(related="task_id.document_ids", string="作业指导")
    task_file_ids = fields.Many2many(related="task_id.file_ids")
    special_process = fields.Selection(related="routing_line_id.special_process")

    next_wo_id = fields.Many2one("roke.work.order", string="下道工单", store=True, compute="_compute_next_work_order")

    @api.depends("task_id")
    def _compute_next_work_order(self):
        for v in self:
            if v.type == "生产":
                sequence = min(v.mapped("sequence"))
                next_wos = v.task_id.work_order_ids.filtered(lambda wo: wo.sequence > sequence)
                next_wo = sorted(next_wos, key=lambda x: x['sequence'], reverse=False)
                if not next_wo:
                    continue
                v.next_wo_id = next_wo[0].id
            elif v.type == "返修":
                v.next_wo_id = v.repair_wr_id.work_order_id.id
            elif v.type == "补件":
                sequence = v.sequence
                next_wos = v.repair_wr_id.scrap_line_ids.scrap_work_order_ids.filtered(lambda wo: wo.sequence > sequence)
                next_wo = sorted(next_wos, key=lambda x: x['sequence'], reverse=False)
                if not next_wo:
                    v.next_wo_id = v.repair_wr_id.work_order_id.id
                else:
                    v.next_wo_id = next_wo[0].id

    def action_assign_work_center(self):
        self.ensure_one()
        # 检查是否是特殊工序 
        # 打开指派向导
        return {
            'name': '指派工作中心',
            'type': 'ir.actions.act_window',
            'res_model': 'assign.work.center.wizard',
            'view_mode': 'form',
            'target': 'new',
            'context': {
                'default_work_order_id': self.id,
            }
        }

    def _get_wo_allow_qty(self):
        if self.is_quality:
            # 质检工单的可报工数取被检工单的数量
            # 待检数=被检工单所有报工数合计
            allow_qty = sum(self.quality_origin_wo_id.record_ids.mapped("finish_qty")) + \
                        sum(self.quality_origin_wo_id.record_ids.mapped("unqualified_qty")) - \
                        self.quality_origin_wo_id.qualified_qty
            default_qty = allow_qty
        else:
            allow_qty, default_qty = super(InheritWorkOrder, self)._get_wo_allow_qty()
        return allow_qty, default_qty

    def _get_bom_transportation_line_vals(self, record, bom):
        """
        获取BOM发料内容
        :return:
        """
        return {
            'product_index': record.product_id.id,
            'partner_id': record.entrust_customer.id,
            'product_id': bom.product_id.id,
            "qty": bom.qty * record.entrust_qty,
            'wo_id': record.id,
            'origin_work_id': record.id
        }

    def confirm_transportation_material(self):
        product_list = []
        # 检验工单是否符合发料条件
        for i in self:
            # 检查自由报工
            if i.env.user.company_id.freedom_work != "允许":
                i._check_freedom_work(i.finish_qty)
            if not i.entrust_customer:
                raise UserError('请选择委外客户!')
            if i.entrust_state == '已发料':
                raise UserError('存在已发料的委外工单,请重新选择!')
        # 检测是否含有委外出库的调拨类型
        type_id = self.env['roke.mes.stock.picking.type'].search([('index', '=', 'WW/OUT/')], limit=1)
        if not type_id:
            raise UserError('请前往仓库管理->作业类型,配置标识为 WW/OUT/ 的出库类型的委外出库!')
        # 数据创建展示
        for record in self:
            for bom in record.routing_line_id.p_bom_ids:
                product_list.append((0, 0, self._get_bom_transportation_line_vals(record, bom)))
        res = self.env["transportation.product.info.wizard"].create({
            "picking_type_id": type_id.id,
            "src_location_id": type_id.src_location_id.id,
            "dest_location_id": type_id.dest_location_id.id,
            "entrust_ids": self.ids,
            "move_line_ids": product_list
        })
        return {
            'name': '发料',
            'type': 'ir.actions.act_window',
            'res_model': 'transportation.product.info.wizard',
            'view_type': 'form',
            'view_mode': 'form',
            'target': 'new',
            'res_id': res.id,
            'context': {}
        }
