from odoo import models, fields, api, _
from odoo.exceptions import ValidationError


class InheritWorkOrder(models.Model):
    _inherit = "roke.work.order"

    document_ids = fields.Many2many(related="task_id.document_ids", string="作业指导")
    task_file_ids = fields.Many2many(related="task_id.file_ids")
    repair_task_id = fields.Many2one('roke.production.task', string="上级任务")
    repair_wr_id = fields.Many2one('roke.work.record', string="原始返修报工单")

    next_wo_id = fields.Many2one("roke.work.order", string="下道工单", store=True, compute="_compute_next_work_order")

    @api.depends("task_id")
    def _compute_next_work_order(self):
        for v in self:
            if v.type == "生产":
                sequence = min(v.mapped("sequence"))
                next_wos = v.task_id.work_order_ids.filtered(lambda wo: wo.sequence > sequence)
                next_wo = sorted(next_wos, key=lambda x: x['sequence'], reverse=False)
                if not next_wo:
                    continue
                v.next_wo_id = next_wo[0].id
            elif v.type == "返修":
                v.next_wo_id = v.repair_wr_id.work_order_id.id
            elif v.type == "补件":
                sequence = v.sequence
                next_wos = v.repair_wr_id.scrap_line_ids.scrap_work_order_ids.filtered(lambda wo: wo.sequence > sequence)
                next_wo = sorted(next_wos, key=lambda x: x['sequence'], reverse=False)
                if not next_wo:
                    v.next_wo_id = v.repair_wr_id.work_order_id.id
                else:
                    v.next_wo_id = next_wo[0].id

    def write(self, vals):
        id_dict = {}
        for v in self:
            id_dict[str(v.id)] = {"state": v.state, "old_finish_qty": v.finish_qty}
        res = super(InheritWorkOrder, self).write(vals)
        for v in self:
            old_dict = id_dict.get(str(v.id), "")
            old_state = old_dict.get("state", "")
            old_finish_qty = old_dict.get("old_finish_qty", 0)
            if "state" not in vals.keys():
                return res
            state = vals.get("state", "")
            if state in ["已完工", "强制完成"]:
                repair_order_id = self.env["roke.repair.order.line"].sudo().search([
                    ("repair_work_order_id", "=", self.id)], limit=1)
                if repair_order_id:
                    repair_order_id.write({
                        "state": "返修完成",
                        "execute_qty": repair_order_id.execute_qty + v.finish_qty
                    })
                    repair_order_id.order_id.write({"state": "返修完成"})
            if state == "未完工" and old_state in ["暂停", "强制完工", "已完工"]:
                repair_order_id = self.env["roke.repair.order.line"].sudo().search([
                    ("repair_work_order_id", "=", self.id)], limit=1)
                if repair_order_id:
                    repair_order_id.write({
                        "state": "返修中",
                        "execute_qty": repair_order_id.execute_qty + v.finish_qty - old_finish_qty
                    })
                    repair_order_id.order_id.write({"state": "返修中"})
        return res

    def _get_production_multiple(self, previous):
        """
        获取当前生产倍数
        :param previous:
        :return:
        """
        judge = "切割" in self.process_id.name or "压铆" in self.process_id.name or "折弯" in self.process_id.name
        if not self.routing_line_id and not judge:
            # 无工艺明细直接返回
            return 1
        if "切割" in self.process_id.name and self.task_id:
            multiple = self.task_id.spare_parts_qty / self.task_id.plan_qty
        elif "压铆" in self.process_id.name and self.task_id:
            multiple = self.task_id.press_riveting_qty / self.task_id.plan_qty
        elif "折弯" in self.process_id.name and self.task_id:
            multiple = self.task_id.bending_qty / self.task_id.plan_qty
        else:
            multiple = self.routing_line_id.multiple
        if len(previous) > 1:
            # 取到最小数对应的前道工单
            rel_previous = self._multi_previous_get_min_finished(previous)
            rel_previous_judge = "切割" in rel_previous.process_id.name or "压铆" in rel_previous.process_id.name \
                                 or "折弯" in rel_previous.process_id.name
            if not rel_previous.routing_line_id.multiple and not rel_previous_judge:  # 无倍数直接返回
                return 1
            if "切割" in rel_previous.process_id.name:
                rel_previous_multiple = rel_previous.task_id.spare_parts_qty / rel_previous.task_id.plan_qty
            elif "压铆" in rel_previous.process_id.name:
                rel_previous_multiple = rel_previous.task_id.press_riveting_qty / rel_previous.task_id.plan_qty
            elif "折弯" in rel_previous.process_id.name:
                rel_previous_multiple = rel_previous.task_id.bending_qty / rel_previous.task_id.plan_qty
            else:
                rel_previous_multiple = rel_previous.routing_line_id.multiple
            return multiple / rel_previous_multiple
        if not previous.routing_line_id.multiple and not judge:  # 无倍数直接返回
            return 1
        elif not previous and judge:
            return 1

        if "切割" in previous.process_id.name and self.task_id and previous.task_id:
            previous_multiple = previous.task_id.spare_parts_qty / self.task_id.plan_qty
        elif "压铆" in previous.process_id.name and self.task_id and previous.task_id:
            previous_multiple = previous.task_id.press_riveting_qty / self.task_id.plan_qty
        elif "折弯" in previous.process_id.name and self.task_id and previous.task_id:
            previous_multiple = previous.task_id.bending_qty / self.task_id.plan_qty
        else:
            previous_multiple = previous.routing_line_id.multiple
        return (multiple / previous_multiple) if previous_multiple else 0


class InheritRokeWorkRecordModel(models.Model):
    _inherit = "roke.work.record"

    def withdraw_work_order(self, work_order):
        if not work_order.task_id and not work_order.record_ids and work_order.type == "生产":
            # 无生产任务且当前工单下无其它报工记录时时，撤回直接删除工单
            work_order.unlink()

    def withdraw(self):
        """
        撤回报工
        :return:
        """
        for v in self:
            for repair in v.repair_line_ids:
                if repair.repair_work_order_id.record_ids:
                    raise ValidationError("返修明细中的返修工单已报工，无法撤销报工！")
                repair.repair_work_order_id.unlink()
            for scrap in v.scrap_line_ids:
                if scrap.scrap_work_order_ids.record_ids:
                    raise ValidationError("返修明细中的返修工单已报工，无法撤销报工！")
                scrap.scrap_work_order_ids.unlink()
        super(InheritRokeWorkRecordModel, self).withdraw()
