# -*- coding: utf-8 -*-
import http.client
import mimetypes
import math
from codecs import encode
from datetime import timedelta

import pymssql
from odoo import models, fields, api, _
from odoo.exceptions import ValidationError
import logging

_logger = logging.getLogger(__name__)


class InheritProductionTask(models.Model):
    _inherit = "roke.production.task"
    _order = "main_task_code desc, code asc"

    file_ids = fields.Many2many("ir.attachment", "roke_production_task_ir_attachment", "pro_id", "att_id")

    work_order_hours = fields.Float(string="总报工时", compute="_compute_work_order_hours")

    task_type = fields.Selection([
        ('main', '主任务'),
        ('sub', '子任务')
    ], string='任务类型', compute='_compute_task_type', store=True)

    main_task_code = fields.Char(string='主任务编号', compute='_compute_task_type', index=True, store=True)

    flowing_water = fields.Char(string="物料流水")

    @api.depends('work_order_ids')
    def _compute_work_order_hours(self):
        for record in self:
            record.work_order_hours = sum(record.work_order_ids.mapped("work_hours"))

    @api.depends('code')
    def _compute_task_type(self):
        for record in self:
            if '-' in (record.code or ""):
                task_str = record.code.split('-')
                if task_str[-1] == "1":
                    record.task_type = 'main'
                    record.main_task_code = task_str[0]
                else:
                    record.task_type = 'sub'
                    record.main_task_code = task_str[0]
            else:
                record.task_type = 'main'
                record.main_task_code = record.code

    def write(self, vals):
        res = super(InheritProductionTask, self).write(vals)
        if vals.get('file_ids'):
            files = []
            for i in self.file_ids:
                a = self.env['documents.document'].create({
                    'attachment_id': i.id,
                    'folder_id': 2})
                files.append(a.id)
                self.document_ids = [(4, a.id)]
        if "finish_qty" in vals.keys():
            system_id = self.env.ref("roke_workstation_sync_ps.roke_workstation_sync_ps_integrate_system")
            sync_address = system_id.sync_address
            sync_port = system_id.sync_port
            sync_uname = system_id.sync_uname
            sync_passwd = system_id.sync_passwd
            db_name = system_id.db_name
            try:
                conn = pymssql.connect(host=sync_address, user=sync_uname, password=sync_passwd, database=db_name)
                cur = conn.cursor()
            except Exception as e:
                raise ValidationError("数据库链接失败，请检查数据库链接参数")

            for v in self:
                sql = f"""
                    UPDATE JHMRP
                    SET JHMRP_CCSL = {vals.get('finish_qty', 0)}
                    WHERE JHMRP_JHBH = '{v.code}'
                """
                try:
                    cur.execute(sql)
                    conn.commit()
                except Exception as e:
                    raise ValidationError("数据库查询失败")
                finally:
                    cur.close()
                    conn.close()
        return res

    def change_routing_id_work_order(self, routing):
        product = self.product_id
        plan_qty = self.plan_qty
        task_type = self.type
        routing_lines = self.get_create_wo_routing_lines(routing)
        work_orders = [(5, 0, 0)]
        date_num = (self.plan_date - self.plan_start_date).days
        work_order_num = len(routing_lines)
        if date_num == 0:
            for routing_line in routing_lines:
                date = self.plan_start_date
                work_order_data = self._get_new_work_order_data(routing_line, product, plan_qty, task_type)
                work_order_data['planned_start_time'] = date
                work_order_data['plan_date'] = date
                work_order_data['priority'] = self.priority
                work_orders.append((0, 0, work_order_data))
        else:
            num = work_order_num // date_num
            if num < 1:
                num = 1
            day = 0
            circulation_num = 1
            for routing_line in routing_lines:
                date = self.plan_start_date + timedelta(days=day)
                if circulation_num == num:
                    circulation_num = 1
                    if day + 1 <= date_num:
                        day += 1
                else:
                    circulation_num += 1
                work_order_data = self._get_new_work_order_data(routing_line, product, plan_qty, task_type)
                work_order_data['planned_start_time'] = date
                work_order_data['plan_date'] = date
                work_order_data['priority'] = self.priority
                work_orders.append((0, 0, work_order_data))
        return work_orders

    def craft_design(self):
        if self.record_ids and not self.routing_id.routing_task_id:
            raise ValidationError("该任务已报工，无法生成新的工艺设计。")
        if not self.routing_id:
            routing_id = self.env["roke.routing"].create({
                "name": f"{self.product_id.name}" or ""
            })
            self.write({"routing_id": routing_id.id})
            self.product_id.write({"routing_id": routing_id.id})
            self.write({
                "work_order_ids": self.change_routing_id_work_order(routing_id)
            })
        elif self.routing_id.state == "确认":
            routing_id = self.routing_id.copy()
            routing_id.update({
                "name": f"{routing_id.name}"
            })
            self.write({"routing_id": routing_id.id})
            self.product_id.write({"routing_id": routing_id.id})
            self.write({
                "work_order_ids": self.change_routing_id_work_order(routing_id)
            })
        else:
            routing_id = self.routing_id
        return {
            "name": f"工艺设计({self.code})",
            "type": "ir.actions.client",
            "tag": "sdddl_project.roke_craft_design",
            "target": "current",
            "params": {
                "controller": f"/roke/craft_design/index?product_id={self.product_id.id}&routing_id={routing_id.id}"
            }
        }

    def create_bom_stock_picking(self):
        """
        BOM领料
        :return:
        """
        self.ensure_one()
        picking_type = self._get_default_picking_type()
        if not picking_type:
            raise ValidationError("未找到默认的调拨类型，请创建业务标识为生产领料单的库存作业类型")
        system_id = self.env.ref("roke_workstation_sync_ps.roke_workstation_sync_ps_integrate_system")
        sync_address = system_id.sync_address
        sync_port = system_id.sync_port
        sync_uname = system_id.sync_uname
        sync_passwd = system_id.sync_passwd
        db_name = system_id.db_name
        try:
            conn = pymssql.connect(host=sync_address, user=sync_uname, password=sync_passwd, database=db_name)
            cur = conn.cursor()
        except Exception as e:
            raise ValidationError("数据库链接失败，请检查数据库链接参数")
        try:
            cur.execute(f"""
                select jswlzd_wlbh as wlbh,(jsbom_zxsl /jsbom_fxsl) as compare
                from JSBOM, jswlzd 
                where JSBOM_FXLS = '{self.flowing_water}'
                and JSBOM_ZXLS =JSWLZD_LSBH
            """)
            rows = cur.fetchall()
            rowdesc = cur.description
        except Exception as e:
            raise ValidationError("数据库查询失败")
        finally:
            cur.close()
            conn.close()
        bom_res = []
        rows_data = [
            dict(zip([col[0] for col in rowdesc], row))
            for row in rows
        ]
        for v in rows_data:
            product = self.env["roke.product"].sudo().search([("code", "=", v.get("wlbh", ""))], limit=1)
            bom_res.append((0, 0, {
                "picking_type_id": picking_type.id,
                "material_id": product.id,
                "demand_qty": self.plan_qty * v.get("compare", 1),
                "qty": self.plan_qty * v.get("compare", 1)
            }))
            _logger.info(f"数据： {product.id}")
        res = self.env["roke.bom.create.picking.wizard"].create({
            "line_ids": bom_res
        })
        return {
            'name': 'bom领料',
            'type': 'ir.actions.act_window',
            'view_mode': 'form',
            'target': 'new',
            'res_id': res.id,
            'res_model': 'roke.bom.create.picking.wizard'
        }
